/*!
 * tp-color-picker.js - Color Picker Plugin for Revolution Slider
 * @version: 2.0 (16.01.2018)
 * @author ThemePunch
*/

;window.RSColor = {

	defaultValue: '#ffffff',
	isColor: /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i,

	get: function(val) {

		if(!val) return 'transparent';
		return RSColor.process(val, true)[0];

	},

	parse: function(val, prop, returnColorType) {

		val = RSColor.process(val, true);
		var ar = [];

		if(!prop) ar[0] = val[0];
		else ar[0] = prop + ': ' + val[0] + ';';

		if(returnColorType) ar[1] = val[1];
		return ar;

	},

	convert: function(color, opacity) {

		// falsey or non-string
		if(!color || typeof color !== 'string') return RSColor.defaultValue;

		// transparent
		if(color === 'transparent') return color;

		// gradients
		if(color.search(/\[\{/) !== -1 || color.search('gradient') !== -1) return RSColor.process(color, true)[0];

		// if no opacity value exists
		if(typeof opacity === 'undefined' || isNaN(opacity)) return RSColor.process(color, true)[0];

		// convert opacity from float to int
		opacity = parseFloat(opacity);
		if(opacity <= 1) opacity *= 100;

		// min/max opacity
		opacity = Math.max(Math.min(parseInt(opacity, 10), 100), 0);

		// transparent for 0 opacity
		if(opacity === 0) return 'transparent';

		try {

			if(color.search('#') !== -1 || color.length < 8) {

				if(!RSColor.isColor.test(color)) color = color.replace(/[^A-Za-z0-9#]/g, '');
				return RSColor.processRgba(RSColor.sanitizeHex(color), opacity);

			}

			else {

				color = RSColor.rgbValues(color, 3);
				return RSColor.rgbaString(color[0], color[1], color[2], opacity * 0.01);

			}

		}
		catch(e) {

			return RSColor.defaultValue;

		}

	},

	process: function(clr, processColor) {

		if(typeof clr !== 'string') {

			if(processColor) RSColor.sanitizeGradient(clr);
			return [RSColor.processGradient(clr), 'gradient', clr];

		}
		else if(clr.trim() === 'transparent') {

			return ['transparent', 'transparent'];

		}
		else if(clr.search(/\[\{/) !== -1) {

			try {

				clr = JSON.parse(clr.replace(/\&/g, '"'));
				if(processColor) RSColor.sanitizeGradient(clr);
				return [RSColor.processGradient(clr), 'gradient', clr];

			}
			catch(e) {

				console.log('RSColor.process() failed to parse JSON string');
				return [

					'linear-gradient(0deg, rgba(255, 255, 255, 1) 0%, rgba(0, 0, 0, 1) 100%)',
					'gradient', {

						'type': 'linear',
						'angle': '0',
						'colors': [

							{'r': '255', 'g': '255', 'b': '255', 'a': '1', 'position': '0', 'align': 'bottom'},
							{'r': '0', 'g': '0', 'b': '0', 'a': '1', 'position': '100', 'align': 'bottom'}

						]
					}
				];
			}

		}
		else if(clr.search('-gradient') !== -1) {

			// gradient was not stored as a JSON string for some reason so just return the color itself
			return [clr, 'gradient_css', RSColor.reverseGradient(clr)];

		}
		else if(clr.search('#') !== -1) {

			return [RSColor.sanitizeHex(clr), 'hex'];

		}
		else if(clr.search('rgba') !== -1) {

			clr = clr.replace(/\s/g, '');
			if(clr.search(/,\)/) !== -1) clr = clr.split(',)')[0] + ',1)';
			return [clr, 'rgba'];

		}
		else {

			return [clr.replace(/\s/g, ''), 'rgb'];

		}

	},

	transparentRgba: function(val, processed) {

		if(!processed) {

			var tpe = RSColor.process(val)[1];
			if(tpe !== 'rgba') return false;

		}

		return RSColor.rgbValues(val, 4)[3] === '0';

	},

	rgbValues: function(values, num) {

		values = values.substring(values.indexOf('(') + 1, values.lastIndexOf(')')).split(',');
		if(values.length === 3 && num === 4) values[3] = '1';

		for(var i = 0; i < num; i++) if(values[i] !== undefined) values[i] = values[i].trim();
		return values;

	},

	rgbString: function(r, g, b) {

		return 'rgb(' + r + ', ' + g + ', ' + b + ')';

	},

	rgbaString: function(r, g, b, a) {

		return 'rgba(' + r + ', ' + g + ', ' + b + ', ' + a + ')';

	},

	rgbToHex: function(clr) {

		var values = RSColor.rgbValues(clr, 3);
		return RSColor.getRgbToHex(values[0], values[1], values[2]);

	},

	rgbaToHex: function(clr) {

		var values = RSColor.rgbValues(clr, 4);
		return [RSColor.getRgbToHex(values[0], values[1], values[2]), values[3]];

	},

	getOpacity: function(val) {

		return parseInt(RSColor.rgbValues(val, 4)[3] * 100, 10) + '%';

	},

	getRgbToHex: function(r, g, b) {

		return '#' + ("0" + parseInt(r, 10).toString(16)).slice(-2) +
					 ("0" + parseInt(g, 10).toString(16)).slice(-2) +
					 ("0" + parseInt(b, 10).toString(16)).slice(-2);

	},

	sanitizeAlpha: function(alpha) {

		return parseFloat(Math.max(Math.min(parseFloat(alpha), 1), 0).toFixed(2).replace(/\.?0*$/, ''));

	},

	joinToRgba:function(val) {

		val = val.split('||');
		return RSColor.convert(val[0], val[1]);

	},

	processRgba: function(hex, opacity) {

		hex = hex.replace('#', '');

		var hasOpacity = typeof opacity !== 'undefined',
		rgb = hasOpacity ? 'rgba' : 'rgb',
		color = rgb + '(' +
				parseInt(hex.substring(0, 2), 16) + ', ' +
				parseInt(hex.substring(2, 4), 16) + ', ' +
				parseInt(hex.substring(4, 6), 16);

		if(hasOpacity) color +=  ', ' + (parseInt(opacity, 10) * 0.01).toFixed(2).replace(/\.?0*$/, '') + ')';
		else color +=  ')';

		return color;

	},

	hexToRGB: function(hex) {

		hex = hex.replace('#', '');

		return [

			parseInt(hex.substring(0, 2), 16),
			parseInt(hex.substring(2, 4), 16),
			parseInt(hex.substring(4, 6), 16)

		];

	},

	processGradient: function(obj, sanitize, preview) {

		if(sanitize) RSColor.sanitizeGradient(obj);
		if(!preview && obj.easing && obj.easing !== 'none') {

			if(typeof jQuery !== 'undefined') obj = RVS.F.safeExtend(true, {}, obj);
			RSColor.easeGradient(obj);

		}

		var angle,
			middle,
			tpe = obj.type,
			begin = tpe + '-gradient(';

		if(tpe === 'linear') {
			angle = parseInt(obj.angle, 10);
			middle = angle !== 180 ? angle + 'deg, ' : '';
		}
		else {
			middle = 'ellipse at center, ';
		}

		var colors = obj.colors,
			len = colors.length,
			end = '',
			clr;

		for(var i = 0; i < len; i++) {

			if(i > 0) end += ', ';
			clr = colors[i];
			end += 'rgba(' + clr.r + ', ' + clr.g + ', ' + clr.b + ', ' + clr.a + ') ' + clr.position + '%';

		}

		return begin + middle + end + ')';

	},

	sanitizeHex: function(hex) {

		hex = hex.replace('#', '').trim();
		if(hex.length === 3) {

			var a = hex.charAt(0),
				b = hex.charAt(1),
				c = hex.charAt(2);

			hex = a + a + b + b + c + c;

		}

		return '#' + hex;

	},

	sortPoints: function(a, b) {

		if(a.hasOwnProperty('x')) return a.x < b.x ? -1 : a.x > b.x ? 1 : 0;
		else return a.position < b.position ? -1 : a.position > b.position ? 1 : 0;

	},

	distColor: function(px, bx, ex, bv, ev) {

		return Math.max(Math.min(Math.round(Math.abs((((px - bx) / (ex - bx)) * (ev - bv)) + bv)), 255), 0);

	},

	distAlpha: function(px, bx, ex, bv, ev) {

		return Math.max(Math.min(Math.abs(parseFloat((((px - bx) / (ex - bx)) * (ev - bv)).toFixed(2)) + parseFloat(bv)), 1), 0);

	},

	sanitizeGradient: function(obj) {

		var colors = obj.colors,
			len = colors.length,
			ar = [],
			prev;

		colors.sort(RSColor.sortPoints);
		for(var i = 0; i < len; i++) {

			var cur = colors[i];
			delete cur.align;

			cur.a = RSColor.sanitizeAlpha(cur.a);
			cur.r = parseInt(cur.r, 10);
			cur.g = parseInt(cur.g, 10);
			cur.b = parseInt(cur.b, 10);
			cur.position = parseInt(cur.position, 10);

			if(prev) {
				if(JSON.stringify(cur) !== JSON.stringify(prev)) ar[ar.length] = cur;
			}
			else {
				ar[ar.length] = cur;
			}

			prev = cur;

		}

		obj.colors = ar;
		return obj;

	},

	// accounting for cases where gradient wasn't saved as a JSON string for some reason
	reverseGradient: function(str) {

		// hsl colors not supported yet
		if(str.search('hsl') !== -1) return str;

		str = str.replace(/\-moz\-|\-webkit\-/, '')
			     .replace('to left', '90deg')
			     .replace('to bottom', '180deg')
			     .replace('to top', '0deg')
			     .replace('to right', '270deg')
				 .replace(';', '');

		var gradient = str.split('-gradient(');
		if(gradient.length < 2) return str;

		var degree = '0',
			grad = gradient[1].trim();

		if(grad.search('ellipse at center') === -1) {

			if(grad.search('deg') !== -1) {

				grad = grad.split('deg');
				degree = grad[0].trim();
				grad = grad[1].trim();

			}

		}
		else {

			grad = grad.replace('ellipse at center', '');

		}

		if(grad.charAt(0) === ',') grad = grad.slice(1);
		if(grad.charAt(grad.length - 1) === ',') grad = grad.slice(0, grad.length - 1);

		var colors = grad.split('%'),
			list = [],
			index = 0,
			perc,
			prev,
			leg,
			tpe,
			clr,
			len;

		colors.pop();
		len = colors.length;

		for(var i = 0; i < len; i++) {

			clr = colors[i].trim();
			leg = clr.length;
			perc = '';

			if(clr.charAt(0) === ',') clr = clr.slice(1);
			if(clr.search(' ') === -1) return str;

			perc = clr.split(' ');
			perc = parseInt(perc[perc.length - 1], 10);

			while(leg--) {

				index = leg;
				if(clr[leg] === ' ') break;

			}

			clr = clr.substring(0, index);
			if(clr.search(/\)/) !== -1) clr = clr.split(')')[0].trim() + ')';
			else clr = clr.split(' ')[0].trim();

			tpe = RSColor.process(clr)[1];
			if(tpe === 'hex') {

				clr = RSColor.sanitizeHex(clr);
				clr = RSColor.processRgba(clr);

			}

			if(prev && prev === clr) continue;
			prev = clr;

			clr = RSColor.rgbValues(clr, 4);
			list[list.length] = {

				r: clr[0],
				g: clr[1],
				b: clr[2],
				a: clr[3],
				position: perc,
				align: 'top'

			};

			list[list.length] = {

				r: clr[0],
				g: clr[1],
				b: clr[2],
				a: 1,
				position: perc,
				align: 'bottom'

			};

		}

		return JSON.stringify({

			type: gradient[0].trim(),
			angle: degree,
			colors: list

		});

	},

	insertPoints: function(start, end, ar, easing, strength) {

		var startPos = start.position,
			endPos = end.position;

		if(startPos > endPos) return;

		var positions = [],
			point,
			val,
			px;

		for(var i = 0; i < strength; i++) {

			val = parseFloat(easing(i, 0, 1, strength).toFixed(2)) * (endPos - startPos) + startPos;
			if(val > startPos && val < endPos) positions[positions.length] = val;

		}

		var len = positions.length,
			count = parseFloat(((endPos - startPos) / (len + 1)).toFixed(2)),
			p = parseFloat(count + startPos),
			startA,
			endA;

		for(i = 0; i < len; i++) {

			px = positions[i];
			if(px === start.position) continue;

			startA = RSColor.sanitizeAlpha(start.a);
			endA = RSColor.sanitizeAlpha(end.a);

			point = {

				position: p,
				r: start.r !== end.r ? Math.round(RSColor.distColor(px, startPos, endPos, start.r, end.r)) : start.r,
				g: start.g !== end.g ? Math.round(RSColor.distColor(px, startPos, endPos, start.g, end.g)) : start.g,
				b: start.b !== end.b ? Math.round(RSColor.distColor(px, startPos, endPos, start.b, end.b)) : start.b,
				a: startA !== endA ? RSColor.sanitizeAlpha(RSColor.distAlpha(px, startPos, endPos, start.a, end.a)) : startA

			};

			p += count;
			p = parseFloat(p.toFixed(2));
			ar[ar.length] = point;

		}

	},

	easeGradient: function(gradient) {

		var strength = (parseInt(gradient.strength, 10) * 0.01) * 15,
			points = gradient.colors,
			easing = gradient.easing,
			len = points.length - 1,
			ar = [];

		points.sort(RSColor.sortPoints);
		easing = easing.split('.');
		easing = RSColor.easing[easing[0]][easing[1]];

		for(var i = 0; i < len; i++) {

			ar[ar.length] = points[i];
			RSColor.insertPoints(points[i], points[i + 1], ar, easing, strength);

		}

		ar[ar.length] = points[len];
		gradient.colors = ar;

	},

	/* http://robertpenner.com/easing_terms_of_use.html */
	easing: {

		quint:{easeout:function(n,t,e,u){return e*((n=n/u-1)*n*n*n*n+1)+t;},easein:function(n,t,e,u){return e*(n/=u)*n*n*n*n+t;},easeinout:function(n,t,e,u){return(n/=u/2)<1?e/2*n*n*n*n*n+t:e/2*((n-=2)*n*n*n*n+2)+t;}},
		quad:{easein:function(n,t,e,u){return e*(n/=u)*n+t;},easeout:function(n,t,e,u){return-e*(n/=u)*(n-2)+t;},easeinout:function(n,t,e,u){return(n/=u/2)<1?e/2*n*n+t:-e/2*(--n*(n-2)-1)+t;}},
		quart:{easein:function(n,t,e,u){return e*(n/=u)*n*n*n+t;},easeout:function(n,t,e,u){return-e*((n=n/u-1)*n*n*n-1)+t;},easeinout:function(n,t,e,u){return(n/=u/2)<1?e/2*n*n*n*n+t:-e/2*((n-=2)*n*n*n-2)+t;}},
		cubic:{easein:function(n,t,e,u){return e*(n/=u)*n*n+t;},easeout:function(n,t,e,u){return e*((n=n/u-1)*n*n+1)+t;},easeinout:function(n,t,e,u){return(n/=u/2)<1?e/2*n*n*n+t:e/2*((n-=2)*n*n+2)+t;}},
		circ:{easein:function(n,t,e,u){return-e*(Math.sqrt(1-(n/=u)*n)-1)+t;},easeout:function(n,t,e,u){return e*Math.sqrt(1-(n=n/u-1)*n)+t;},easeinout:function(n,t,e,u){return(n/=u/2)<1?-e/2*(Math.sqrt(1-n*n)-1)+t:e/2*(Math.sqrt(1-(n-=2)*n)+1)+t;}},
		sine:{easein:function(n,t,e,u){return-e*Math.cos(n/u*(Math.PI/2))+e+t;},easeout:function(n,t,e,u){return e*Math.sin(n/u*(Math.PI/2))+t;},easeinout:function(n,t,e,u){return-e/2*(Math.cos(Math.PI*n/u)-1)+t;}},
		expo:{easein:function(n,t,e,u){return 0===n?t:e*Math.pow(2,10*(n/u-1))+t;},easeout:function(n,t,e,u){return n===u?t+e:e*(1-Math.pow(2,-10*n/u))+t;},easeinout:function(n,t,e,u){return 0===n?t:n===u?t+e:(n/=u/2)<1?e/2*Math.pow(2,10*(n-1))+t:e/2*(2-Math.pow(2,-10*--n))+t;}},
		bounce:{easeout:function (t,b,c,d) {if((t/=d)<(1/2.75)){return c*(7.5625*t*t)+b;}else if (t<(2/2.75)){return c*(7.5625*(t-=(1.5/2.75))*t+0.75)+b;}else if (t<(2.5/2.75)) {return c*(7.5625*(t-=(2.25/2.75))*t + 0.9375) + b;} else {return c*(7.5625*(t-=(2.625/2.75))*t+0.984375)+b;}},easein: function(t,b,c,d) {return c-RSColor.easing.bounce.easeout(d-t,0,c,d)+b;},easeinout: function(t,b,c,d){if(t<d/2)return RSColor.easing.bounce.easein(t*2,0,c,d)*0.5+b;return RSColor.easing.bounce.easeout(t*2-d,0,c,d)*0.5+c*0.5+b;}}

	}


};

(function($) {

	RVS.DOC = RVS.DOC===undefined ? jQuery(document) : RVS.DOC;

	var angle,
		inited,
		bodies,
		isFull,
		points,
		radial,
		easing,
		hitTop,
		onEdit,
		onInit,
		onAjax,
		colorR,
		colorG,
		colorB,
		colorR2,
		colorG2,
		colorB2,
		cPicker,
		prepped,
		reverse,
		gradBtn,
		gradHex,
		inFocus,
		defEdit,
		defAjax,
		strength,
		colorBtn,
		colorHex,
		pointLoc,
		gradIris,
		curPoint,
		appended,
		gradCore,
		onCancel,
		defTitle,
		defCancel,
		defChange,
		dragPoint,
		topPoints,
		botPoints,
		hitBottom,
		textareas,
		curSquare,
		curCorner,
		colorIris,
		langColor,
		gradInput,
		onReverse,
		wheelDown,
		colorView,
		radiusHor,
		radiusVer,
		easingRow,
		radiusRad,
		customLang,
		defRefresh,
		angleWheel,
		wheelPoint,
		gradOutput,
		colorClear,
		directions,
		pointIndex,
		storedValue,
		storedColor,
		gradViewed,
		gradOpacity,
		groupPoints,
		pointerWrap,
		colorDelete,
		forceChange,
		curLanguage,
		customAdded,
		defWidgetId,
		selectedHex,
		globalColors,
		globalColorMode,
		globalColorActive,
		globalCandidates,
		origSettings,
		opacityFixer,
		previewColor,
		currentInput,
		presetGroups,
		colorOpacity,
		openingValue,
		openingColor,
		appendedHTML,
		currentColor,
		supressColor,
		supressWheel,
		supressCheck,
		supressClick,
		gradientsOpen,
		isTransparent,
		mainContainer,
		selectedColor,
		opacitySlider,
		supressOpacity,
		defaultClasses,
		changeCallback,
		previewGradient,
		gradientPreview = {};

	var hitWidth = 236,
		maxPoints = 20,
		centerWheel = 25,
		presetRows = 3,
		presetColumns = 4,
		deleteBuffer = 10,
		warningBuffer = 15,
		sliderHeight = 150,
		presetMargin = 10,
		defMode = 'full',
		dragObj = {axis: "x", containment: "#revbuilder-cpicker-point-wrap"};

	var lang = {
		'color': 'Color',
		'solid_color': 'Solid',
		'gradient_color': 'Gradient',
		'gradient_easing': 'Gradient Easing',
		'easing_strength': 'Easing/Strength',
		'currently_editing': 'Currently Editing',
		'core_presets': 'Core Presets',
		'custom_presets': 'Custom Presets',
		'enter_a_name': 'Enter a Name',
		'save_a_new_preset': 'Save a new Preset',
		'save': 'Save Preset',
		'color_hex_value': 'Hex',
		'opacity': 'Opacity',
		'clear': 'Clear Color',
		'location': 'Location',
		'delete': 'Delete Marker',
		'horizontal': 'Horizontal',
		'vertical': 'Vertical',
		'radial': 'Radial',
		'enter_angle': 'Enter Angle',
		'reverse_gradient': 'Reverse Gradient',
		'delete_confirm': 'Remove/Delete this custom preset color?',
		'naming_error': 'Please enter a unique name for the new color preset',
		'preset_collection': 'Preset Collection',
		'color_type' : 'Color Type',
		'close':'Close',
		'save_exit':'Save',
		'apply':'Apply',
		'rgb': 'RGB',
		'preset_group': 'Preset Group',
		'save_preset': 'Save Preset',
		'pos_opacity': 'Pos / Opacity'

	};

	var defColors = [

		'#FFFFFF',
		'#000000',
		'#FF3A2D',
		'#007AFF',
		'#4CD964',
		'#FFCC00',
		'#C7C7CC',
		'#8E8E93',
		'#FFD3E0',
		'#34AADC',
		'#E0F8D8',
		'#5AC8FA',
		'#5856D6',
		'#FF69B4',
		'#800000',
		'#009933'

	];

	var defGradients = [

		{'0': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:247,&g&:247,&b&:247,&a&:&1&,&position&:0,&align&:&top&},{&r&:247,&g&:247,&b&:247,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:215,&g&:215,&b&:215,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:215,&g&:215,&b&:215,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'1': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:74,&g&:74,&b&:74,&a&:&1&,&position&:0,&align&:&top&},{&r&:74,&g&:74,&b&:74,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:43,&g&:43,&b&:43,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:43,&g&:43,&b&:43,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'2': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:219,&g&:221,&b&:222,&a&:&1&,&position&:0,&align&:&top&},{&r&:219,&g&:221,&b&:222,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:137,&g&:140,&b&:144,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:137,&g&:140,&b&:144,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'3': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:26,&g&:214,&b&:253,&a&:&1&,&position&:0,&align&:&top&},{&r&:26,&g&:214,&b&:253,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:29,&g&:98,&b&:240,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:29,&g&:98,&b&:240,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'4': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:198,&g&:68,&b&:252,&a&:&1&,&position&:0,&align&:&top&},{&r&:198,&g&:68,&b&:252,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:88,&g&:86,&b&:214,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:88,&g&:86,&b&:214,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'5': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:255,&g&:94,&b&:58,&a&:&1&,&position&:0,&align&:&top&},{&r&:255,&g&:94,&b&:58,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:255,&g&:42,&b&:104,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:42,&b&:104,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'6': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:228,&g&:221,&b&:202,&a&:&1&,&position&:0,&align&:&top&},{&r&:228,&g&:221,&b&:202,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:214,&g&:206,&b&:195,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:214,&g&:206,&b&:195,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'7': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:255,&g&:219,&b&:76,&a&:&1&,&position&:0,&align&:&top&},{&r&:255,&g&:219,&b&:76,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:255,&g&:205,&b&:2,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:205,&b&:2,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'8': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:255,&g&:149,&b&:0,&a&:&1&,&position&:0,&align&:&top&},{&r&:255,&g&:149,&b&:0,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:255,&g&:94,&b&:58,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:94,&b&:58,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'9': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:82,&g&:237,&b&:199,&a&:&1&,&position&:0,&align&:&top&},{&r&:82,&g&:237,&b&:199,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:90,&g&:200,&b&:251,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:90,&g&:200,&b&:251,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'10': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:228,&g&:183,&b&:240,&a&:&1&,&position&:0,&align&:&top&},{&r&:228,&g&:183,&b&:240,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:200,&g&:110,&b&:223,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:200,&g&:110,&b&:223,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'11': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:135,&g&:252,&b&:112,&a&:&1&,&position&:0,&align&:&top&},{&r&:135,&g&:252,&b&:112,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:11,&g&:211,&b&:24,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:11,&g&:211,&b&:24,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'12': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:61,&g&:78,&b&:129,&a&:&1&,&position&:0,&align&:&top&},{&r&:61,&g&:78,&b&:129,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:87,&g&:83,&b&:201,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:110,&g&:127,&b&:243,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:110,&g&:127,&b&:243,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'13': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:35,&g&:21,&b&:87,&a&:&1&,&position&:0,&align&:&top&},{&r&:35,&g&:21,&b&:87,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:68,&g&:16,&b&:122,&a&:&1&,&position&:29,&align&:&bottom&},{&r&:255,&g&:19,&b&:97,&a&:&1&,&position&:67,&align&:&bottom&},{&r&:255,&g&:248,&b&:0,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:248,&b&:0,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'14': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:105,&g&:234,&b&:203,&a&:&1&,&position&:0,&align&:&top&},{&r&:105,&g&:234,&b&:203,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:234,&g&:204,&b&:248,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:102,&g&:84,&b&:241,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:102,&g&:84,&b&:241,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'15': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:255,&g&:5,&b&:124,&a&:&1&,&position&:0,&align&:&top&},{&r&:255,&g&:5,&b&:124,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:124,&g&:100,&b&:213,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:76,&g&:195,&b&:255,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:76,&g&:195,&b&:255,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'16': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:255,&g&:5,&b&:124,&a&:&1&,&position&:0,&align&:&top&},{&r&:255,&g&:5,&b&:124,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:141,&g&:11,&b&:147,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:50,&g&:21,&b&:117,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:50,&g&:21,&b&:117,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'17': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:164,&g&:69,&b&:178,&a&:&1&,&position&:0,&align&:&top&},{&r&:164,&g&:69,&b&:178,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:212,&g&:24,&b&:114,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:255,&g&:0,&b&:102,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:0,&b&:102,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'18': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:158,&g&:251,&b&:211,&a&:&1&,&position&:0,&align&:&top&},{&r&:158,&g&:251,&b&:211,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:87,&g&:233,&b&:242,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:69,&g&:212,&b&:251,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:69,&g&:212,&b&:251,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'19': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:172,&g&:50,&b&:228,&a&:&1&,&position&:0,&align&:&top&},{&r&:172,&g&:50,&b&:228,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:121,&g&:24,&b&:242,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:72,&g&:1,&b&:255,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:72,&g&:1,&b&:255,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'20': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:112,&g&:133,&b&:182,&a&:&1&,&position&:0,&align&:&top&},{&r&:112,&g&:133,&b&:182,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:135,&g&:167,&b&:217,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:222,&g&:243,&b&:248,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:222,&g&:243,&b&:248,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'21': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:34,&g&:225,&b&:255,&a&:&1&,&position&:0,&align&:&top&},{&r&:34,&g&:225,&b&:255,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:29,&g&:143,&b&:225,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:98,&g&:94,&b&:177,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:98,&g&:94,&b&:177,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'22': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:44,&g&:216,&b&:213,&a&:&1&,&position&:0,&align&:&top&},{&r&:44,&g&:216,&b&:213,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:107,&g&:141,&b&:214,&a&:&1&,&position&:50,&align&:&bottom&},{&r&:142,&g&:55,&b&:215,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:142,&g&:55,&b&:215,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'23': '{&type&:&linear&,&angle&:&160&,&colors&:[{&r&:44,&g&:216,&b&:213,&a&:&1&,&position&:0,&align&:&top&},{&r&:44,&g&:216,&b&:213,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:197,&g&:193,&b&:255,&a&:&1&,&position&:56,&align&:&bottom&},{&r&:255,&g&:186,&b&:195,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:186,&b&:195,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'24': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:191,&g&:217,&b&:254,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:191,&g&:217,&b&:254,&a&:&1&,&position&:0,&align&:&top&},{&r&:223,&g&:137,&b&:181,&a&:&1&,&position&:100,&align&:&top&},{&r&:223,&g&:137,&b&:181,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'25': '{&type&:&linear&,&angle&:&340&,&colors&:[{&r&:97,&g&:97,&b&:97,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:97,&g&:97,&b&:97,&a&:&1&,&position&:0,&align&:&top&},{&r&:155,&g&:197,&b&:195,&a&:&1&,&position&:100,&align&:&top&},{&r&:155,&g&:197,&b&:195,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'26': '{&type&:&linear&,&angle&:&90&,&colors&:[{&r&:36,&g&:57,&b&:73,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:36,&g&:57,&b&:73,&a&:&1&,&position&:0,&align&:&top&},{&r&:81,&g&:127,&b&:164,&a&:&1&,&position&:100,&align&:&top&},{&r&:81,&g&:127,&b&:164,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'27': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:234,&g&:205,&b&:163,&a&:&1&,&position&:0,&align&:&top&},{&r&:234,&g&:205,&b&:163,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:230,&g&:185,&b&:128,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:230,&g&:185,&b&:128,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'28': '{&type&:&linear&,&angle&:&45&,&colors&:[{&r&:238,&g&:156,&b&:167,&a&:&1&,&position&:0,&align&:&top&},{&r&:238,&g&:156,&b&:167,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:255,&g&:221,&b&:225,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:255,&g&:221,&b&:225,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'29': '{&type&:&linear&,&angle&:&340&,&colors&:[{&r&:247,&g&:148,&b&:164,&a&:&1&,&position&:0,&align&:&top&},{&r&:247,&g&:148,&b&:164,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:253,&g&:214,&b&:189,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:253,&g&:214,&b&:189,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'30': '{&type&:&linear&,&angle&:&45&,&colors&:[{&r&:135,&g&:77,&b&:162,&a&:&1&,&position&:0,&align&:&top&},{&r&:135,&g&:77,&b&:162,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:196,&g&:58,&b&:48,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:196,&g&:58,&b&:48,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'31': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:243,&g&:231,&b&:233,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:243,&g&:231,&b&:233,&a&:&1&,&position&:0,&align&:&top&},{&r&:218,&g&:212,&b&:236,&a&:&1&,&position&:100,&align&:&top&},{&r&:218,&g&:212,&b&:236,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'32': '{&type&:&linear&,&angle&:&320&,&colors&:[{&r&:43,&g&:88,&b&:118,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:43,&g&:88,&b&:118,&a&:&1&,&position&:0,&align&:&top&},{&r&:78,&g&:67,&b&:118,&a&:&1&,&position&:100,&align&:&top&},{&r&:78,&g&:67,&b&:118,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'33': '{&type&:&linear&,&angle&:&60&,&colors&:[{&r&:41,&g&:50,&b&:60,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:41,&g&:50,&b&:60,&a&:&1&,&position&:0,&align&:&top&},{&r&:72,&g&:85,&b&:99,&a&:&1&,&position&:100,&align&:&top&},{&r&:72,&g&:85,&b&:99,&a&:&1&,&position&:100,&align&:&bottom&}]}'},
		{'34': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:233,&g&:233,&b&:231,&a&:&1&,&position&:0,&align&:&top&},{&r&:233,&g&:233,&b&:231,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:239,&g&:238,&b&:236,&a&:&1&,&position&:25,&align&:&bottom&},{&r&:238,&g&:238,&b&:238,&a&:&1&,&position&:70,&align&:&bottom&},{&r&:213,&g&:212,&b&:208,&a&:&1&,&position&:100,&align&:&bottom&},{&r&:213,&g&:212,&b&:208,&a&:&1&,&position&:100,&align&:&top&}]}'},
		{'35': '{&type&:&linear&,&angle&:&180&,&colors&:[{&r&:251,&g&:200,&b&:212,&a&:&1&,&position&:0,&align&:&bottom&},{&r&:251,&g&:200,&b&:212,&a&:&1&,&position&:0,&align&:&top&},{&r&:151,&g&:149,&b&:240,&a&:&1&,&position&:100,&align&:&top&},{&r&:151,&g&:149,&b&:240,&a&:&1&,&position&:100,&align&:&bottom&}]}'}

	];

	var markup =

	'<div id="revbuilder-cpicker-back" class="revbuilder-cpicker-close"></div>' +

	'<div id="revbuilder-cpicker">' +

		'<div id="revbuilder-cpicker-container">' +

			'<div id="revbuilder-cpicker-left" class="revbuilder-cpicker-column">' +

				'<div id="revbuilder-cpicker-head">' +

					'<div id="revbuilder-cpicker-drag" class="revbuilder-cpicker-draggable"></div>' +

					'<div id="revbuilder-cpicker-type-wrap" class="revbuilder-cpicker-side revbuilder-cpicker-left revbuilder-cpicker-candrag">' +
						'<span data-text="color_type" class="revbuilder-cpicker-candrag revbuilder-cpicker-title"></span>' +
						'<div id="revbuilder-cpicker-btn-wrap">' +
							'<span id="revbuilder-cpicker-color-btn" class="revbuilder-cpicker-main-btn"><i class="material-icons">invert_colors</i></span>' +
							'<span id="revbuilder-cpicker-gradient-btn" class="revbuilder-cpicker-main-btn"><i class="material-icons">gradient</i></span>' +
						'</div>' +
					'</div>' +

					'<div id="revbuilder-cpicker-editing" class="revbuilder-cpicker-candrag">' +
						'<span id="revbuilder-cpicker-edit-title" data-text="currently_editing"></span>' +
						'<span id="revbuilder-cpicker-current-edit" class="revbuilder-cpicker-txt-space-left revbuilder-cpicker-bright"></span>' +
					'</div>' +

				'</div>' +

				'<div id="revbuilder-cpicker-base">' +

					'<div id="revbuilder-cpicker-colors">' +

						'<div>' +

							'<div class="revbuilder-cpicker-side revbuilder-cpicker-left">' +

								'<div class="revbuilder-cpicker-row first">' +

									'<span data-text="color" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<span id="revbuilder-cpicker-preview-color" class="revbuilder-cpicker-preview"></span>' +
										'<input type="text" id="revbuilder-cpicker-color-hex" class="revbuilder-cpicker-input revbuilder-cpicker-hex" value="#ffffff" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span data-text="rgb" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small first">' +
										'<input type="text" id="revbuilder-cpicker-r" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
									'</div>' +

									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small">' +
										'<input type="text" id="revbuilder-cpicker-g" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
									'</div>' +

									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small">' +
										'<input type="text" id="revbuilder-cpicker-b" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span data-text="opacity" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<input type="text" id="revbuilder-cpicker-color-opacity" class="revbuilder-cpicker-input revbuilder-cpicker-opacity-input" value="100%" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +
									'<span class="revbuilder-cpicker-spacer"></span>' +
									'<span id="revbuilder-cpciker-clear-hex" class="revbuilder-cpicker-btn revbuilder-cpicker-hover"><i class="material-icons">delete</i><span data-text="clear"></span></span>' +
								'</div>' +

							'</div>' +

							'<div class="revbuilder-cpicker-side">' +

								'<div class="revbuilder-cpicker-iris">' +

									'<input id="revbuilder-cpicker-iris-color" class="revbuilder-cpicker-iris-input" />' +

									'<div id="revbuilder-cpicker-scroller" class="iris-slider iris-strip">' +
										'<div id="revbuilder-cpicker-scroll-bg"></div>' +
										'<div id="revbuilder-cpicker-scroll" class="iris-slider-offset"></div>' +
									'</div>' +

								'</div>' +

							'</div>' +

						'</div>' +

						'<div id="revbuilder-cpicker-color-presets" class="revbuilder-cpicker-presets">' +

							'<div class="revbuilder-cpicker-side revbuilder-cpicker-left">' +

								'<div class="revbuilder-cpicker-row first">' +

									'<span data-text="preset_group" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<select id="revbuilder-cpicker-preset-colors" class="revbuilder-cpicker-dropdown revbuilder-cpicker-preset-selector">' +
											'<option value="core">Defaults</option>' +
											'<option value="custom">Custom</option>' +
										'</select>' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span data-text="save_preset" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<input type="text" class="revbuilder-cpicker-input revbuilder-cpicker-preset-save" data-placeholder="enter_a_name" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +
									'<span class="revbuilder-cpicker-spacer"></span>' +
									'<span class="revbuilder-cpicker-btn revbuilder-cpicker-hover revbuilder-cpicker-save-preset-btn" data-alert="naming_error" data-message="Please enter a unique name for the new color preset">' +
										'<i class="material-icons">save</i><span data-text="save"></span>' +
									'</span>' +
								'</div>' +

							'</div>' +

							'<div class="revbuilder-cpicker-side">' +

								'<div id="revbuilder-cpicker-colors-core" class="revbuilder-cpicker-presets-group ps-container"></div>' +
								'<div id="revbuilder-cpicker-colors-custom" class="revbuilder-cpicker-presets-group ps-container revbuilder-cpicker-presets-custom"></div>' +

							'</div>' +

						'</div>' +

					'</div>' +

					'<div id="revbuilder-cpicker-gradients">' +

						'<div>' +

							'<div class="revbuilder-cpicker-side revbuilder-cpicker-left revbuilder-cpicker-supress">' +

								'<div class="revbuilder-cpicker-row first">' +

									'<div class="revbuilder-cpicker-gradient-block">' +

										'<div id="revbuilder-cpicker-gradient-input" class="revbuilder-cpicker-gradient-builder">' +

											'<span id="revbuilder-cpicker-hit-top" class="revbuilder-cpicker-builder-hit"></span>' +
											'<div id="revbuilder-cpicker-point-wrap">' +
												'<div id="revbuilder-cpciker-point-container"></div>' +
											'</div>' +
											'<span id="revbuilder-cpicker-hit-bottom" class="revbuilder-cpicker-builder-hit"></span>' +

										'</div>' +

									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span data-text="pos_opacity" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-basic revbuilder-cpicker-supress first disabled">' +
										'<input type="text" id="revbuilder-cpicker-color-location" class="revbuilder-cpicker-input" value="100%" maxlength="4" />' +
									'</div>' +

									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-basic revbuilder-cpicker-supress disabled">' +
										'<input type="text" id="revbuilder-cpicker-grad-opacity" class="revbuilder-cpicker-input revbuilder-cpicker-opacity-input" value="100%" maxlength="4" />' +
									'</div>' +

									'<span id="revbuilder-cpicker-color-delete" class="revbuilder-cpicker-btn revbuilder-cpicker-btn-small revbuilder-cpicker-hover revbuilder-cpicker-point-delete revbuilder-cpicker-supress"><i class="material-icons">delete</i></span>' +
									'<div class="clearfix"></div>' +

								'</div>' +

								'<div id="revpicker-easing-row" class="revbuilder-cpicker-row">' +

									'<span id="revbuilder-cpicker-easing-title" data-text="gradient_easing" class="revbuilder-cpicker-title"></span>' +
									'<span id="revbuilder-cpicker-strength-title" data-text="easing_strength" class="revbuilder-cpicker-title"></span>' +

									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<select id="revbuilder-cpicker-easing" class="revbuilder-cpicker-dropdown">' +
											'<option value="none">None</option>' +
											'<option value="sine.easeout">sine.out</option>' +
											'<option value="sine.easein">sine.in</option>' +
											'<option value="sine.easeinout">sine.inOut</option>' +
											'<option value="quad.easeout">Quad.easeOut</option>' +
											'<option value="quad.easein">Quad.easeIn</option>' +
											'<option value="quad.easeinout">Quad.easeInOut</option>' +
											'<option value="cubic.easeout">Cubic.easeOut</option>' +
											'<option value="cubic.easein">Cubic.easeIn</option>' +
											'<option value="cubic.easeinout">Cubic.easeInOut</option>' +
											'<option value="circ.easeout">circ.out</option>' +
											'<option value="circ.easein">circ.in</option>' +
											'<option value="circ.easeinout">circ.inOut</option>' +
											'<option value="quart.easeout">Quart.easeOut</option>' +
											'<option value="quart.easein">Quart.easeIn</option>' +
											'<option value="quart.easeinout">Quart.easeInOut</option>' +
											'<option value="quint.easeout">Quint.easeOut</option>' +
											'<option value="quint.easein">Quint.easeIn</option>' +
											'<option value="quint.easeinout">Quint.easeInOut</option>' +
											'<option value="expo.easeout">expo.out</option>' +
											'<option value="expo.easein">expo.in</option>' +
											'<option value="expo.easeinout">expo.inOut</option>' +
											'<option value="bounce.easeout">bounce.out</option>' +
											'<option value="bounce.easein">bounce.in</option>' +
											'<option value="bounce.easeinout">bounce.inOut</option>' +
										'</select>' +

									'</div>' +

									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-basic revbuilder-cpicker-supress revbuilder-cpicker-easing-strength">' +
										'<input type="text" id="revbuilder-cpicker-strength" class="revbuilder-cpicker-input" value="100%" maxlength="5" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-hideable">' +

									'<div class="revbuilder-cpicker-row extended">' +

										'<span data-text="color" class="revbuilder-cpicker-title"></span>' +
										'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-supress">' +
											'<span id="revbuilder-cpicker-preview-gradient" class="revbuilder-cpicker-preview"></span>' +
											'<input type="text" id="revbuilder-cpicker-gradient-hex" class="revbuilder-cpicker-input revbuilder-cpicker-hex" value="#ffffff" />' +
										'</div>' +

									'</div>' +

									'<div class="revbuilder-cpicker-row">' +

										'<span data-text="rgb" class="revbuilder-cpicker-title"></span>' +
										'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small revbuilder-cpicker-supress first">' +
											'<input type="text" id="revbuilder-cpicker-r2" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
										'</div>' +

										'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small revbuilder-cpicker-supress">' +
											'<input type="text" id="revbuilder-cpicker-g2" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
										'</div>' +

										'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small revbuilder-cpicker-supress">' +
											'<input type="text" id="revbuilder-cpicker-b2" class="revbuilder-cpicker-input revbuilder-cpicker-rgb" value="255" />' +
										'</div>' +

									'</div>' +

									'<div class="revbuilder-cpicker-row">' +

										'<span class="revbuilder-cpicker-spacer"></span>' +
										'<span id="revbuilder-cpciker-clear-gradient" class="revbuilder-cpicker-btn revbuilder-cpicker-hover revbuilder-cpicker-supress"><i class="material-icons">delete</i><span data-text="clear"></span></span>' +

									'</div>' +

								'</div>' +

							'</div>' +

							'<div class="revbuilder-cpicker-side">' +

								'<div class="revbuilder-cpicker-row first">' +

									'<div class="revbuilder-cpicker-gradient-block">' +
										'<div id="revbuilder-cpicker-gradient-output" class="revbuilder-cpicker-gradient-builder"></div>' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span id="revbuilder-cpicker-orientation-horizontal" class="revbuilder-cpicker-direction revbuilder-cpicker-orientation revbuilder-cpicker-supress selected" data-direction="90">' +
										'<span class="revbuilder-cpicker-preset-tile"></span>' +
										'<span id="revbuilder-cpicker-direction-horizontal" class="revbuilder-cpicker-preset-bg"></span>' +
									'</span>' +

									'<span id="revbuilder-cpicker-orientation-vertical" class="revbuilder-cpicker-direction revbuilder-cpicker-orientation revbuilder-cpicker-supress" data-direction="180">' +
										'<span class="revbuilder-cpicker-preset-tile"></span>' +
										'<span id="revbuilder-cpicker-direction-vertical" class="revbuilder-cpicker-preset-bg"></span>' +
									'</span>' +

									'<span id="revbuilder-cpicker-orientation-radial" class="revbuilder-cpicker-direction revbuilder-cpicker-orientation revbuilder-cpicker-supress" data-direction="radial">' +
										'<span class="revbuilder-cpicker-preset-tile"></span>' +
										'<span id="revbuilder-cpicker-direction-radial" class="revbuilder-cpicker-preset-bg"></span>' +
									'</span>' +

									'<div id="revbuilder-cpicker-wheel" class="revbuilder-cpicker-supress">' +
										'<div id="revbuilder-cpicker-wheel-inner"><span id="revbuilder-cpicker-wheel-point"></span></div>' +
									'</div>' +

									'<div id="revbuilder-cpicker-radial-wrap" class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover revbuilder-cpicker-input-small revbuilder-cpicker-supress">' +
										'<input type="text" class="revbuilder-cpicker-input" id="revbuilder-cpicker-meta-angle" maxlength="4">' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row revbuilder-cpicker-no-margin">' +

									'<span data-text="reverse_gradient" class="revbuilder-cpicker-title revbuilder-cpicker-label"></span>' +
									'<div id="revbuilder-cpicker-meta-reverse" class="revbuilder-cpicker-onoff-wrap revbuilder-cpicker-supress">' +
										'<div class="revbuilder-cpicker-onoff-inner">' +
											'<div class="revbuilder-cpicker-onoff revbuilder-cpicker-off"><span>OFF</span></div>' +
											'<div class="revbuilder-cpicker-onoff revbuilder-cpicker-on"><span>ON</span></div>' +
										'</div>' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-hideable">' +

									'<div class="revbuilder-cpicker-row">' +

										'<div class="revbuilder-cpicker-iris revbuilder-cpicker-supress">' +
											'<input id="revbuilder-cpicker-iris-gradient" class="revbuilder-cpicker-iris-input" />' +
										'</div>' +

									'</div>' +

								'</div>' +

							'</div>' +

						'</div>' +

						'<div id="revbuilder-cpicker-gradient-presets" class="revbuilder-cpicker-presets">' +

							'<div class="revbuilder-cpicker-side revbuilder-cpicker-left">' +

								'<div class="revbuilder-cpicker-row first">' +

									'<span data-text="preset_group" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<select id="revbuilder-cpicker-preset-gradients" class="revbuilder-cpicker-dropdown revbuilder-cpicker-preset-selector">' +
											'<option value="core">Defaults</option>' +
											'<option value="custom">Custom</option>' +
										'</select>' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +

									'<span data-text="save_preset" class="revbuilder-cpicker-title"></span>' +
									'<div class="revbuilder-cpicker-input-wrap revbuilder-cpicker-hover">' +
										'<input type="text" class="revbuilder-cpicker-input revbuilder-cpicker-preset-save" data-placeholder="enter_a_name" />' +
									'</div>' +

								'</div>' +

								'<div class="revbuilder-cpicker-row">' +
									'<span class="revbuilder-cpicker-spacer"></span>' +
									'<span class="revbuilder-cpicker-btn revbuilder-cpicker-hover revbuilder-cpicker-save-preset-btn" data-alert="naming_error" data-message="Please enter a unique name for the new color preset">' +
										'<i class="material-icons">save</i><span data-text="save"></span>' +
									'</span>' +
								'</div>' +

							'</div>' +

							'<div class="revbuilder-cpicker-side">' +

								'<div id="revbuilder-cpicker-gradients-core" class="revbuilder-cpicker-presets-group ps-container"></div>' +
								'<div id="revbuilder-cpicker-gradients-custom" class="revbuilder-cpicker-presets-group ps-container revbuilder-cpicker-presets-custom"></div>' +

							'</div>' +

						'</div>' +

					'</div>' +

				'</div>' +

			'</div>' +

			'<div id="revbuilder-cpicker-right" class="revbuilder-cpicker-column">' +

				'<div id="revbuilder-cpicker-exit" class="revbuilder-cpicker-close revbuilder-cpicker-btn-large revbuilder-cpicker-hover"><i class="material-icons revbuilder-cpicker-hover">close</i></div>' +

				'<div id="revbuilder-cpicker-toggle">' +
					'<div id="revbuilder-cpicker-palette" class="revbuilder-cpicker-btn-large revbuilder-cpicker-hover"><i class="material-icons revbuilder-cpicker-hover revbuilder-cpicker-palette-closed">palette</i><i class="material-icons revbuilder-cpicker-hover revbuilder-cpicker-palette-open">keyboard_arrow_up</i></div>' +
					'<div id="revbuilder-cpicker-check" class="revbuilder-cpicker-btn-large"><i class="material-icons revbuilder-cpicker-hover">check</i></div>' +
				'</div>' +

			'</div>' +

		'</div>' +

		'<span id="revbuilder-cpicker-remove-delete" data-text="delete_confirm"></span>' +

	'</div>';

	function changeDegree(grad, deg) {

		var begin;
		if(grad.search('deg') === -1) {

			grad = grad.split('(');
			begin = grad[0];
			grad.shift();
			grad = begin + '(180deg, ' + grad.join('(');

		}

		grad = grad.split('(');

		begin = grad[0];
		grad.shift();

		var end = grad.join('(').split(',');
		end.shift();

		deg = typeof deg !== 'undefined' ? deg + 'deg,' : 'ellipse at center,';
		return begin + '(' + deg + end.join(',');

	}

	function gradientView(val) {

		return changeDegree(val.replace('radial-', 'linear-').replace('ellipse at center,', ''), '90');

	}

	function replaceText() {

		this.innerHTML = curLanguage[getAttribute(this, 'data-text')];

	}

	function replaceHolder() {

		this.setAttribute('placeholder', curLanguage[getAttribute(this, 'data-placeholder')]);

	}

	function replaceAlert() {

		this.setAttribute('data-message', curLanguage[getAttribute(this, 'data-alert')]);

	}

	function getAttribute(el, attr) {

		return el.getAttribute(attr) || '';

	}

	function writeLanguage(language) {

		if(!language) language = {};
		if(typeof language === 'string') language = JSON.parse(language.replace(/\&/g, '"'));

		curLanguage = $.extend({}, lang, language);
		langColor = curLanguage.color;

		cPicker.find('*[data-placeholder]').each(replaceHolder);
		cPicker.find('*[data-alert]').each(replaceAlert);
		cPicker.find('*[data-text]').each(replaceText);

	}

	function newPreset(val, core, cls, grad) {

		var titl,
			obj,
			el;

		if(!$.isPlainObject(val)) {

			titl = val;

		}
		else {

			var angl,
				tpe;

			for(var prop in val) {

				if(!val.hasOwnProperty(prop)) continue;

				val = val[prop];

				if(typeof val === 'string') {

					val = RSColor.process(val);
					if(val[1] === 'gradient') {

						obj = val[2];
						angl = obj.angle;
						tpe = obj.type;
						val = RSColor.processGradient($.extend(true, {}, obj), true);

					}
					else {
						val = val[0];
					}

				}
				else {

					angl = val.angle;
					tpe = val.type;

				}

				if(!isNaN(prop)) {
					titl = tpe !== 'radial' ? angl + '&deg;' : 'radial';
				}
				else {
					titl = prop.replace(/_/g, ' ').replace(/\b\w/g, function(chr) {return chr.toUpperCase();});
				}

			}

		}

		if(val !== 'blank') {

			var datas;
			if($.isPlainObject(val)) {

				obj = val;
				datas = '';
				val = grad || RSColor.processGradient($.extend(true, {}, val), true);

			}

			var shell = '<span class="revbuilder-cpicker-color tptip' + cls + '" data-title="' + titl + '" data-color="' + val + '">' +
					    '<span class="revbuilder-cpicker-preset-tile"></span>' +
					    '<span class="revbuilder-cpicker-preset-bg" style="background: ' + val + '"></span>';

			if(!core) shell += '<span class="revbuilder-cpicker-delete"><i class="material-icons">clear</i></span>';
			shell += '</span>';

			el = $(shell);
			if(obj) el.data('gradient', obj);
			return el[0];

		}
		else {

			el = document.createElement('span');
			el.className = 'revbuilder-cpicker-color blank';
			return el;

		}

	}

	function checkPreset() {

		var presetColor = getAttribute(this, 'data-color').toLowerCase(),
		toCheck = !supressCheck ? presetColor === openingColor.toLowerCase() : false;

		if(presetColor === selectedHex || toCheck) {

			var $this = $(this);

			$this.closest('.revbuilder-cpicker-presets-group').find('.revbuilder-cpicker-color.selected').removeClass('selected');
			selectedColor = $this;

			if(supressCheck && !colorView) setValue(selectedColor.data('gradient'), true);
			$this.addClass('selected');

			return false;

		}

	}

	function writePresets(container, colors) {

		var frag = document.createDocumentFragment(),
			core = container.search('core') !== -1,
			cls = core ? '' : ' revbuilder-picker-color-custom',
			len = colors.length,
			rows = Math.max(Math.ceil(len / presetColumns), presetRows);

		for(var i = 0; i < rows; i++) {

			while(colors.length < ((i + 1) * presetColumns)) colors[colors.length] = 'blank';

		}

		len = colors.length;
		for(i = 0; i < len; i++) {

			frag.appendChild(newPreset(colors[i], core, cls));

		}

		return ['revbuilder-cpicker-' + container, frag];

	}

	function onChange(gradient, color, trans) {

		if(!currentColor) return;
		if(!gradient) {

			var hex = color || colorHex.val(),
				opacity = typeof trans !== 'undefined' ? trans : colorOpacity.val();

			if(hex === 'transparent') {color = 'transparent';}
			else if(opacity === '100%') {

				color = RSColor.sanitizeHex(hex);

			}
			else {

				color = RSColor.processRgba(hex, opacity);

			}

		}

		var isTrans = color === 'transparent',
			val = !isTrans ? color : '';

		if(!gradient) colorBtn.data('state', color);
		else gradBtn.data('state', color);

		if(!isTrans) {
			currentColor[0].style.background = val;
			previewColor.css('background', val);
		}
		else {
			currentColor.css('background', val);
			previewColor.css('background', val);
		}

		if(onEdit) onEdit(currentInput, color, globalColors, globalColorActive);
		RVS.DOC.trigger('revcolorpickerupdate', [currentInput, color]);

	}

	function setValue(val, fromPreset) {

		var obj = RSColor.process(val),
			type = obj[1],
			clr = obj[0];

		if(isFull) reverse.removeClass('checked');
		if(type !== 'gradient' && type !== 'gradient_css') {

			switch(type) {

				case 'hex':

					val = RSColor.sanitizeHex(clr);
					colorOpacity.val('100%');
					updateSlider(100);

				break;

				case 'rgba':

					var values = RSColor.rgbaToHex(clr),
					opacity = parseInt(values[1] * 100, 10);
					val = values[0];

					colorOpacity.val(opacity + '%');
					updateSlider(opacity);

				break;

				case 'rgb':

					val = RSColor.rgbToHex(clr);
					colorOpacity.val('100%');
					updateSlider(100);

				break;

				default:

					colorClear.trigger('click');
					colorBtn.trigger('click');

				// end default

			}

			colorIris.val(val).change();
			if(!fromPreset) colorBtn.trigger('click');

		}
		else {

			if(isFull) {

				if(type === 'gradient_css') obj[2] = JSON.parse(obj[2]);
				buildGradientInput(obj[2]);
				buildGradientOutput();

				if(!fromPreset) {

					gradViewed = true;
					gradBtn.trigger('click');

				}

			}
			else {

				colorIris.val(RSColor.defaultValue).change();
				colorBtn.trigger('click');

			}

		}

		return [clr, type];

	}

	function getSibling(align, x) {

		var locations = points.slice(),
			len = locations.length,
			pnt;

		locations.sort(RSColor.sortPoints);
		while(len--) {

			pnt = locations[len];
			if(pnt.align === align && pnt.x < x) return pnt;

		}

		len = locations.length;
		for(var i = 0; i < len; i++) {

			pnt = locations[i];
			if(pnt.align === align && pnt.x > x) return pnt;

		}

	}

	function clonePoint(align, pos) {

		var sibling = getSibling(align, pos),
			clr = sibling.color,
			bg = getBgVal(clr, align, true),
			rgb = getRgbVal(clr, true);

		var pnt = newPoint(align, pos, rgb, bg);
		if(curPoint) curPoint.removeClass('active');

		curPoint = $(pnt).addClass('active').appendTo(pointerWrap).draggable(dragObj);
		curSquare = curPoint.children('.revbuilder-cpicker-point-square')[0];
		curCorner = curPoint.children('.revbuilder-cpicker-point-triangle')[0];
		groupPoints = pointerWrap.children();

		var grad = activatePoint(pos);
		buildGradientOutput(pnt);

		if(align === 'bottom') gradIris.val(grad[1]).change();

	}

	function activatePoint(pointX) {

		if(typeof pointX === 'undefined') pointX = points[pointIndex].x;

		var color = curPoint.attr('data-color'),
			bottom = curPoint.hasClass('revbuilder-cpicker-point-bottom');

		if(bottom) {

			if(colorDelete.hasClass('active')) colorDelete.removeClass('active');

			color = RSColor.rgbaToHex(color)[0];
			previewGradient.css('background', color);

			if(cPicker.find('.revbuilder-cpicker-point-bottom').length > 2) {
				colorDelete.addClass('active');
			}

			gradOpacity.parent().addClass('disabled');
			cPicker.addClass('open');

		}
		else {

			if(colorDelete.hasClass('active')) {

				previewGradient.css('background', '');
				colorDelete.removeClass('active');

			}

			var opacity = RSColor.getOpacity(color);
			gradOpacity.attr('data-opacity', opacity).val(opacity).parent().removeClass('disabled');

			if(cPicker.find('.revbuilder-cpicker-point-top').length > 2) {
				colorDelete.addClass('active');
			}

			cPicker.removeClass('open');

		}

		pointLoc.val(pointX + '%').parent().removeClass('disabled');
		return [bottom, color];

	}

	function getBgVal(clr, align, cloned) {

		if(align === 'bottom') return 'rgb(' + clr.r + ',' + clr.g + ',' + clr.b + ')';
		var opacity = !cloned ? clr.a : '1';
		return 'rgba(0, 0, 0, ' + opacity + ')';

	}

	function getRgbVal(clr, cloned) {

		var opacity = !cloned ? clr.a : '1';
		return 'rgba(' + clr.r + ',' + clr.g + ',' + clr.b + ',' + opacity + ')';

	}

	function updateColor(val, gradient) {

		var rgb = RSColor.hexToRGB(val);

		if(!gradient) {

			colorHex.val(val);
			colorR.value = rgb[0];
			colorG.value = rgb[1];
			colorB.value = rgb[2];

			if(colorOpacity[0].value === '0%') colorOpacity.val('100%').trigger('focusout');

		}
		else {

			gradHex.val(val);
			colorR2.value = rgb[0];
			colorG2.value = rgb[1];
			colorB2.value = rgb[2];

		}

		colorHex.rsColorPicker('refresh');

	}

	function setPoints(colors) {

		var frag = document.createDocumentFragment(),
			len = colors.length,
			align,
			clr;

		for(var i = 0; i < len; i++) {

			clr = colors[i];
			align = clr.align;
			frag.appendChild(

				newPoint(

					align,
					clr.position,
					getRgbVal(clr),
					getBgVal(clr, align)

				)

			);

		}

		if(groupPoints) groupPoints.draggable('destroy');
		pointerWrap.empty().append(frag);
		groupPoints = pointerWrap.children().draggable(dragObj);

	}

	function newPoint(align, pos, val, bg) {

		var el = document.createElement('span');
		el.className = 'revbuilder-cpicker-point revbuilder-cpicker-point-' + align;

		if(typeof val === 'string') el.setAttribute('data-color', val);
		else $(el).data('gradient', val);

		el.setAttribute('data-location', pos);
		el.style.left = pos + '%';

		if(align === 'bottom') {

			el.innerHTML =
			'<span class="revbuilder-cpicker-point-triangle" style="border-bottom-color: ' + bg + '"></span>' +
			'<span class="revbuilder-cpicker-point-square" style="background: ' + bg  +'"></span>';

		}
		else {

			el.innerHTML =
			'<span class="revbuilder-cpicker-point-square" style="background: ' + bg + '"></span>' +
			'<span class="revbuilder-cpicker-point-triangle" style="border-top-color: ' + bg + '"></span>';

		}

		return el;

	}

	function getDegree(val) {

		if(!textareas) textareas = document.createElement('textarea');

		if(!val || val === 'radial') val = '0';
		textareas.innerHTML = val + '&deg;';
		return textareas.value;

	}

	function deactivate() {

		if(curPoint) {

			curPoint.removeClass('active');
			curPoint = false;

		}

		pointLoc.parent().addClass('disabled');
		gradOpacity.parent().addClass('disabled');

		colorDelete.removeClass('active');
		colorDelete.removeClass('active');

		previewGradient.css('background', '');
		cPicker.removeClass('open');

	}

	function onClose(e, changed) {

		cPicker.removeClass('active is-basic').hide();
		bodies.removeClass('revbuilder-colorpicker-open');
		mainContainer.css({left: '', top: ''});

		if(appended) {

			appended.remove();
			appended = false;

		}

		if(selectedColor) {

			if(selectedColor.hasClass('selected')) {

				if(changed) currentInput.data('hex', selectedColor.attr('data-color').toLowerCase());
				selectedColor.removeClass('selected');

			}
			else {

				currentInput.removeData('hex');

			}

			selectedColor = false;

		}
		else {

			currentInput.removeData('hex');

		}

		if(!changed) {
			if(onCancel) onCancel(currentInput, storedColor, storedValue, globalColors, globalColorActive);
			if(openingValue && openingValue !== 'transparent') {
				currentColor[0].style.background = openingValue;
				previewColor.css('background', openingValue);
			}
			else {
				currentColor.css('background', '');
				previewColor.css('background', '');
			}

			RVS.DOC.trigger('revcolorpickercancel', [currentInput, storedColor, storedValue]);
		}

		killVars();

	}

	function buildGradientInput(obj) {

		var degree = obj.angle,
			ease = obj.easing ? obj.easing : 'none',
			power = obj.hasOwnProperty('strength') ? obj.strength : '100';

		if(obj.type === 'radial') degree = 'radial';

		directions.removeClass('selected');
		$('.revbuilder-cpicker-orientation[data-direction="' + degree + '"]').addClass('selected');

		setEasingRow(ease);
		easing.val(ease).ddTP('change');

		strength.val(power + '%');
		angle.val(getDegree(degree));

		setEasingRow();
		updateWheel(degree);
		setPoints(obj.colors);

	}

	function gradientColor(preview) {

		var gradient = $.extend(true, {}, gradientPreview);
		return RSColor.processGradient(gradient, true, preview);

	}

	function buildGradientOutput(el, revrse, save, supress) {

		onReverse = revrse;
		calculatePoints();
		onReverse = false;

		var elm,
			color,
			point,
			ar = [],
			gradient,
			len = points.length;

		for(var i = 0; i < len; i++) {

			point = points[i];
			color = point.color;
			ar[i] = color;

			elm = point.el;
			elm.setAttribute('data-color', RSColor.rgbaString(color.r, color.g, color.b, color.a));
			elm.setAttribute('data-opacity', color.a * 100);

			if(el && el === elm) pointIndex = i;

		}

		if(!radial.hasClass('selected')) {

			gradientPreview.type = 'linear';
			gradientPreview.angle = parseInt(angle.val(), 10).toString();

		}
		else {

			gradientPreview.type = 'radial';
			gradientPreview.angle = '0';

		}

		gradientPreview.colors = ar;
		gradientPreview.easing = easing.val();
		gradientPreview.strength = parseInt(strength.val(), 10);

		if(selectedColor && !supressClick) selectedColor.removeClass('selected');
		if(!supress) {

			gradient = $.extend(true, {}, gradientPreview);
			onChange(true, RSColor.processGradient(gradient, true));

		}

		if(!save) {

			if(!radiusHor) radiusHor = document.getElementById('revbuilder-cpicker-direction-horizontal');
			if(!radiusVer) radiusVer = document.getElementById('revbuilder-cpicker-direction-vertical');
			if(!radiusRad) radiusRad = document.getElementById('revbuilder-cpicker-direction-radial');
			if(!gradInput) gradInput = document.getElementById('revbuilder-cpicker-gradient-input');

			gradient = gradientColor(true);
			gradInput.style.background = gradientView(gradient);

			if(gradientPreview.easing !== 'none') gradient = gradientColor();
			gradOutput.style.background = gradient;

			gradientPreview.type = 'linear';
			gradientPreview.angle = '90';
			radiusHor.style.background = gradientColor();

			gradientPreview.angle = '180';
			radiusVer.style.background = gradientColor();

			gradientPreview.type = 'radial';
			gradientPreview.angle = '0';
			radiusRad.style.background = gradientColor();

		}

	}

	function back(i, align) {

		if(i === 0) return false;

		var pnt;
		while(i--) {

			pnt = points[i];
			if(pnt.align !== align) return pnt;

		}

		return false;

	}

	function front(i, align, len) {

		if(i === len) return false;

		var pnt;
		while(i++ < len) {

			pnt = points[i];
			if(pnt.align !== align) return pnt;

		}

		return false;

	}

	function adjustAlpha(point, begin, end) {

		var val,
			beginAlpha = begin.alpha,
			endAlpha = end.alpha;

		if(beginAlpha !== endAlpha) {

			val = RSColor.distAlpha(point.x, begin.x, end.x, beginAlpha, endAlpha).toFixed(2);

		}
		else {

			val = beginAlpha;

		}

		val = RSColor.sanitizeAlpha(val);
		point.alpha = val;
		point.color.a = val;

	}

	function adjustColor(point, begin, end) {

		var p = point.color,
			bv = begin.color,
			ev = end.color;

		if(begin !== end) {

			var px = point.x,
				bx = begin.x,
				ex = end.x;

			p.r = RSColor.distColor(px, bx, ex, bv.r, ev.r);
			p.g = RSColor.distColor(px, bx, ex, bv.g, ev.g);
			p.b = RSColor.distColor(px, bx, ex, bv.b, ev.b);

		}
		else {

			p.r = bv.r;
			p.g = bv.g;
			p.b = bv.b;

		}

	}

	function calculatePoints() {

		points = [];
		topPoints = [];
		botPoints = [];

		groupPoints.each(getPoints);
		points.sort(RSColor.sortPoints);

		var len = points.length,
			iLen = len - 1,
			point,
			begin,
			tpe,
			end;

		for(var i = 0; i < len; i++) {

			point = points[i];
			tpe = point.align;

			begin = back(i, tpe);
			if(begin === false) begin = front(i, tpe, iLen);

			end = front(i, tpe, iLen);
			if(end === false) end = back(i, tpe);

			if(tpe === 'bottom') adjustAlpha(point, begin, end);
			else adjustColor(point, begin, end);

		}

		points.sort(RSColor.sortPoints);

	}

	function getPoints(i) {

		var rgba = RSColor.rgbValues(getAttribute(this, 'data-color'), 4),
			align = this.className.search('bottom') !== -1 ? 'bottom' : 'top',
			alpha = rgba[3].replace(/\.?0*$/, '') || 0,
			pos = this.style.left;

		if(pos.search('px') !== -1) {
			pos = parseInt(pos, 10);
			pos = parseInt((Math.round(pos) / (hitWidth - 2)).toFixed(2) * 100, 10);
		}
		else {
			pos = parseInt(pos, 10);
		}

		if(onReverse) {

			if(pos < 50) pos += (50 - pos) * 2;
			else pos -= (pos - 50) * 2;

		}

		this.style.left = pos + '%';
		this.setAttribute('data-location', pos);

		alpha = RSColor.sanitizeAlpha(alpha);
		points[i] = {

			el: this,
			x: pos,
			alpha: alpha,
			align: align,
			color: {

				r: parseInt(rgba[0], 10),
				g: parseInt(rgba[1], 10),
				b: parseInt(rgba[2], 10),
				a: alpha,
				position: pos,
				align: align

			}

		};

		if(curPoint && curPoint[0] !== this) {

			if(align === 'bottom') botPoints[botPoints.length] = pos;
			else topPoints[topPoints.length] = pos;

		}

	}

	function updateWheel(val) {

		if(!wheelPoint) wheelPoint = document.getElementById('revbuilder-cpicker-wheel-point');

		val = typeof val !== 'undefined' ? val : parseInt(angle.val(), 10);
		wheelPoint.style.transform = 'rotate(' + val + 'deg)';

	}

	function onAngleChange(e, dir, wheel) {

		var isWheel = typeof wheel !== 'undefined',
			val = isWheel ? wheel : parseInt(angle.val(), 10),
			changed,
			value;

		if(e && e.type === 'keyup') {

			changed = !isNaN(val) && val >= -360 && val <= 360;
			value = val;

		}
		else {

			var oValue = parseInt(angle.data('orig-value'), 10);

			if(!val) val = '0';
			if(isNaN(val) || val < -360 || val > 360) {

				val = value = oValue;
				angle.val(getDegree(val));
				changed = true;

			}

			if(val !== oValue) {

				value = val;
				changed = true;
				angle.val(getDegree(val));

				if(!isWheel) {

					val = dir || val;
					directions.removeClass('selected');
					$('.revbuilder-cpicker-orientation[data-direction="' + val + '"]').addClass('selected');

				}

			}

		}

		if(changed || dir) {

			if(value) updateWheel(value);
			buildGradientOutput();

		}

	}

	function onDrag(e, ui) {

		var pos = parseInt((Math.round(ui.position.left) / (hitWidth - 2)).toFixed(2) * 100, 10);
		pointLoc.val(pos + '%').trigger('keyup');

	}

	function onDragStart() {

		var $this = $(this);
		dragPoint = $this.hasClass('revbuilder-cpicker-point-bottom') ? 'bottom' : 'top';
		$this.trigger('click');

	}

	function onDragStop() {

		pointLoc.trigger('focusout');

	}

	function updateSlider(opacity) {

		supressOpacity = true;
		opacitySlider.slider('value', Math.round((opacity * 0.01) * sliderHeight));
		supressOpacity = false;

	}

	function wheelMove(e) {

		var offset = angleWheel.offset(),
			posX = e.pageX - offset.left,
			posY = e.pageY - offset.top;

		if(isNaN(posX) || isNaN(posY)) return;

		var val = Math.atan2(posY - centerWheel, posX - centerWheel) * (180 / Math.PI) + 90;
		if(val < 0) val += 360;

		val = Math.max(0, Math.min(360, Math.round(val)));
		val = 5 * (Math.round(val / 5));

		supressWheel = true;
		onAngleChange(false, false, val);
		supressWheel = false;

	}

	function focusPatch(e) {

		e.stopImmediatePropagation();

	}

	function updatePresets(group) {

		if(!group) group = presetGroups;
		group.each(function() {

			var presets = $(this).find('.revbuilder-cpicker-color').css('margin-bottom', 0),
				len = presets.length;

			presets.each(function(i) {

				if(i < len - 4) this.style.marginBottom = presetMargin + 'px';

			});

		}).RSScroll('update');

	}

	function setEasingRow(val) {

		if(!easingRow) easingRow = $('#revpicker-easing-row');

		if(!val) val = easing.val();
		var method = val === 'none' ? 'removeClass' : 'addClass';
		easingRow[method]('active');

	}

	function writeVars() {

		gradOutput = document.getElementById('revbuilder-cpicker-gradient-output');
		colorR2 = document.getElementById('revbuilder-cpicker-r2');
		colorG2 = document.getElementById('revbuilder-cpicker-g2');
		colorB2 = document.getElementById('revbuilder-cpicker-b2');
		colorR = document.getElementById('revbuilder-cpicker-r');
		colorG = document.getElementById('revbuilder-cpicker-g');
		colorB = document.getElementById('revbuilder-cpicker-b');
		pointerWrap = $('#revbuilder-cpciker-point-container');
		previewColor = $('#revbuilder-cpicker-preview-color');
		colorOpacity = $('#revbuilder-cpicker-color-opacity');
		radial = $('#revbuilder-cpicker-orientation-radial');
		gradCore = $('#revbuilder-cpicker-gradients-core');
		gradHex = $('#revbuilder-cpicker-gradient-hex');
		hitBottom = $('#revbuilder-cpicker-hit-bottom');
		colorHex = $('#revbuilder-cpicker-color-hex');
		hitTop = $('#revbuilder-cpicker-hit-top');

	}

	function killVars() {

		hitTop = null;
		radial = null;
		colorR = null;
		colorG = null;
		colorB = null;
		colorR2 = null;
		colorG2 = null;
		colorB2 = null;
		gradHex = null;
		gradCore = null;
		colorHex = null;
		easingRow = null;
		hitBottom = null;
		radiusHor = null;
		radiusVer = null;
		radiusRad = null;
		gradInput = null;
		textareas = null;
		gradOutput = null;
		wheelPoint = null;
		pointerWrap = null;
		previewColor = null;
		colorOpacity = null;
		currentColor = null;
		currentInput = null;

	}

	function init() {

		if(!prepped) onStart();

		previewGradient = $('#revbuilder-cpicker-preview-gradient');
		colorDelete = $('#revbuilder-cpicker-color-delete');
		gradOpacity = $('#revbuilder-cpicker-grad-opacity');
		pointLoc = $('#revbuilder-cpicker-color-location');
		directions = $('.revbuilder-cpicker-orientation');
		gradIris = $('#revbuilder-cpicker-iris-gradient');
		colorIris = $('#revbuilder-cpicker-iris-color');
		gradBtn = $('#revbuilder-cpicker-gradient-btn');
		colorClear = $('#revbuilder-cpciker-clear-hex');
		reverse = $('#revbuilder-cpicker-meta-reverse');
		opacitySlider = $('#revbuilder-cpicker-scroll');
		colorBtn = $('#revbuilder-cpicker-color-btn');
		strength = $('#revbuilder-cpicker-strength');
		angle = $('#revbuilder-cpicker-meta-angle');
		angleWheel = $('#revbuilder-cpicker-wheel');
		mainContainer = $('#revbuilder-cpicker');
		easing = $('#revbuilder-cpicker-easing');

		dragObj.drag = onDrag;
		dragObj.stop = onDragStop;
		dragObj.start = onDragStart;

		colorBtn.data('state', $('#revbuilder-cpicker-colors').find('.revbuilder-cpicker-color').eq(0).attr('data-color') || '#ffffff');
		gradBtn.data('state', $('#revbuilder-cpicker-gradients').find('.revbuilder-cpicker-color').eq(0).attr('data-color') || 'linear-gradient(0deg, rgba(255, 255, 255, 1) 0%, rgba(0, 0, 0, 1) 100%)');

		mainContainer.draggable({containment: 'window', handle: '.revbuilder-cpicker-draggable', stop: function() {

			mainContainer.css('height', 'auto');

		}});


		presetGroups = $('.revbuilder-cpicker-presets-group').RSScroll({wheelPropagation: false, suppressScrollX: true});

		angleWheel.on('mousedown.revcpicker', function(e) {

			directions.removeClass('selected');
			wheelDown = true;
			wheelMove(e);

		}).on('mousemove.revcpicker', function(e) {

			if(!wheelDown) return;
			wheelMove(e);

		}).on('mouseleave.revcpicker mouseup.revcpicker', function() {

			wheelDown = false;

		});

		$('.revbuilder-cpicker-main-btn').on('click.revcpicker', function() {

			var $this,
				state;

			colorView = this.id.search('gradient') === -1;


			if(currentColor) {

				$this = $(this);
				state = $this.data('state');

			}

			if(colorView) {

				if(currentColor) selectedHex = colorHex.val();
				cPicker.removeClass('gradient-view').addClass('color-view');

			}
			else {

				if(currentColor) selectedHex = state;
				cPicker.removeClass('color-view').addClass('gradient-view');
				if(!gradViewed) {

					gradCore.children('.revbuilder-cpicker-color').not('.blank').eq(0).trigger('click');
					gradViewed = true;

				}

			}

			if(state) {

				var isTrans = state === 'transparent',
					val = !isTrans ? state : '';

				currentColor.css('background', val);
				previewColor.css('background', val);

				supressCheck = true;
				$('.revbuilder-cpicker-color').not('.blank').each(checkPreset);
				supressCheck = false;

				RVS.DOC.trigger('revcolorpickerupdate', [currentInput, state]);

			}

			if(!colorView) {

				supressClick = true;
				$('.revbuilder-cpicker-point-bottom').first().trigger('click');
				supressClick = false;

			}

		});

		$('#revbuilder-cpicker-palette').on('click', function() {

			if(cPicker.hasClass('palette-open')) {

				cPicker.removeClass('palette-open');

			}
			else {

				cPicker.addClass('palette-open');
				updatePresets();

			}

		});

		// save function
		$('#revbuilder-cpicker-check').on('click.revcipicker', function() {

			var ar,
				clr,
				color,
				changed;

			if(cPicker.hasClass('color-view')) {

				var hex = colorHex.val(),
					opacity = colorOpacity.val();

				if(hex === 'transparent') color = 'transparent';
				else if(opacity === '100%') color = RSColor.sanitizeHex(hex);
				else color = RSColor.processRgba(hex, opacity);

				ar = [currentInput, color, false];
				clr = color;

			}
			else {

				deactivate();
				buildGradientOutput(false, false, true, true);
				var grad = JSON.stringify(gradientPreview).replace(/\"/g, '&');
				clr = RSColor.processGradient($.extend(true, {}, gradientPreview), true);

				ar = [currentInput, clr, grad];

			}

			changed = ar[1] !== openingValue;
			if(changed===true || forceChange===true) {
				currentInput.attr('data-color', ar[1]);
				RVS.DOC.trigger('revcolorpickerchanged', ar);
				if(changeCallback) changeCallback(ar[0], ar[1], ar[2], globalColors, globalColorActive);

			}

			onClose(false, changed===true || forceChange===true);

		});

		cPicker.on('click.revcpicker', function(e) {

			var $this = $(e.target);
			if(!$this.closest('#revbuilder-cpicker').length) deactivate();

		});

		$('.revbuilder-cpicker-close').on('click.revcpicker', onClose);

		colorIris.wpColorPicker({palettes: false, width: 190, border: false, hide: false, change: function(e, ui) {

			var val = ui.color.toString();
			this.value = val;

			updateColor(val);

			if(!supressColor) {

				var opacity = colorOpacity.val();
				if(parseInt(opacity, 10) === 0) val = 'transparent';
				onChange(false, val, opacity);

				if(selectedColor) {

					selectedColor.removeClass('selected');
					selectedColor = false;

				}

			}

		}});

		gradIris.wpColorPicker({palettes: false, height: 190,  border: false, hide: false, change: function(e, ui) {

			var val = ui.color.toString();
			this.value = val;

			updateColor(val, true);
			previewGradient.css('background', val);

			curSquare.style.backgroundColor = val;
			curCorner.style.borderBottomColor = val;

			var rgba = RSColor.processRgba(val, 100),
				parsed = RSColor.rgbValues(rgba, 4),
				color = gradientPreview.colors[pointIndex];

			color.r = parsed[0];
			color.g = parsed[1];
			color.b = parsed[2];
			color.a = RSColor.sanitizeAlpha(parsed[3]);

			if(curPoint) curPoint.attr('data-color', rgba);
			buildGradientOutput();

		}});

		opacitySlider.slider({orientation: 'vertical', max: sliderHeight, value: sliderHeight,

			start: function() {

				isTransparent = colorHex.val() === 'transparent';

			},

			slide: function(e, ui) {

				if(!supressOpacity) {

					var opacity = parseInt((ui.value / sliderHeight).toFixed(2) * 100, 10),
						val;

					if(isTransparent) {

						val = opacity ? '#ffffff' : 'transparent';
						colorHex.val(val);

					}

					if(opacity === 0) val = 'transparent';
					onChange(false, val, opacity || 'transparent');
					colorOpacity.val(opacity + '%');

				}

			}

		});

		$('body').on('click.revcpicker', '.revbuilder-cpicker-point', function() {

			pointerWrap.find('.revbuilder-cpicker-point.active').removeClass('active');
			curPoint = $(this).addClass('active');

			curSquare = curPoint.children('.revbuilder-cpicker-point-square')[0];
			curCorner = curPoint.children('.revbuilder-cpicker-point-triangle')[0];

			buildGradientOutput(this);
			var grad = activatePoint();
			if(grad[0]) gradIris.val(grad[1]).change();

		}).on('mousedown.revcpicker', '.revbuilder-cpicker-point', function(e) {

			curPoint = $(this).data('mousestart', e.pageY);

		}).on('mousemove.revcpicker', function(e) {

			if(!curPoint || !curPoint.data('mousestart')) return;

			var start = curPoint.data('mousestart'),
				posY = e.pageY;

			if(curPoint.hasClass('revbuilder-cpicker-point-bottom')) {

				if(posY > start && posY - start > warningBuffer && colorDelete.hasClass('active')) {
					curPoint.addClass('warning');
				}
				else {
					curPoint.removeClass('warning');
				}

			}
			else {

				if(start > posY && start - posY > warningBuffer && colorDelete.hasClass('active')) {
					curPoint.addClass('warning');
				}
				else {
					curPoint.removeClass('warning');
				}

			}

		}).on('mouseup.revcpicker', function(e) {

			if(!curPoint || !curPoint.data('mousestart')) return;

			var start = curPoint.data('mousestart'),
				end = e.pageY;

			curPoint.removeData('mousestart');
			if(curPoint.hasClass('revbuilder-cpicker-point-bottom')) {

				if(end > start && end - start > deleteBuffer && colorDelete.hasClass('active')) {
					colorDelete.trigger('click');
				}
				else {
					curPoint.removeClass('warning');
				}

			}
			else {

				if(start > end && start - end > deleteBuffer && colorDelete.hasClass('active')) {
					colorDelete.trigger('click');
				}
				else {
					curPoint.removeClass('warning');
				}

			}

		}).on('keypress.revcpicker', function(e) {

			if(cPicker.hasClass('active')) {

				var key = e.which;
				if(key == 27) {

					onClose();

				}
				else if(key == 13 && inFocus) {

					inFocus.blur();

				}

			}

		}).on('click.revcpicker', '.revbuilder-cpicker-color:not(.blank)', function() {

			if(selectedColor) {

				if(selectedColor[0] === this && selectedColor.hasClass('selected')) return;
				selectedColor.removeClass('selected');

			}

			selectedColor = $(this);
			if(cPicker.hasClass('color-view')) {

				var val = selectedColor.attr('data-color'),
					tpe = RSColor.process(val),
					opac;

				if(tpe[1] === 'rgba') {

					val = RSColor.rgbaToHex(tpe[0]);
					opac = parseFloat(val[1]) * 100;
					colorOpacity.val(opac + '%').trigger('keyup');
					val = val[0];

				}
				else if(tpe[1] !== 'transparent' && val !== 'transparent') {
					colorOpacity.val('100%').trigger('keyup');
				}
				else {
					supressColor = true;
					colorClear.trigger('click');
				}

				supressColor = true;
				colorIris.val(val).change();
				if(colorHex.val() === 'transparent') colorHex.val(val.toLowerCase());
				supressColor = false;

				var opacity = colorOpacity.val();
				if(parseInt(opacity, 10) === 0) val = 'transparent';

				onChange(false, val, opacity);
				selectedColor.addClass('selected');

			}
			else {

				hitTop.removeClass('full');
				hitBottom.removeClass('full');

				setValue(selectedColor.data('gradient'), true);
				reverse.removeClass('checked');
				gradCore.find('.revbuilder-cpicker-color.selected').removeClass('selected');

				$('.revbuilder-cpicker-point-bottom').first().trigger('click');
				selectedColor.addClass('selected');

			}

		}).on('mouseover.revcpicker', '.revbuilder-cpicker-color:not(.blank)', function() {

			if(gradientsOpen) gradOutput.style.background = getAttribute(this, 'data-color');

		}).on('mouseout.revcpicker', '.revbuilder-cpicker-color:not(.blank)', function() {

			if(gradientsOpen) buildGradientOutput();

		}).on('click.revcpicker', '.revbuilder-cpicker-delete', function() {

			if(!onAjax) {

				console.log('Ajax callback not defined');
				return;

			}

			if(window.confirm(document.getElementById('revbuilder-cpicker-remove-delete').innerHTML)) {

				cPicker.addClass('onajax onajaxdelete');

				var $this = $(this),
					colr = $this.parent(),
					titl = colr.attr('data-title') || '';

				if(!titl) {

					console.log('Preset does not have a name/title');
					return;

				}

				var group = $this.closest('.revbuilder-cpicker-presets-group'),
					ids = group[0].id,
					tpe = ids.search('colors') !== -1 ? 'colors' : 'gradients';

				RVS.DOC.off('revcpicker_onajax_delete.revcpicker').on('revcpicker_onajax_delete.revcpicker', function(evt, err) {

					if(err) console.log(err);

					var group = $this.closest('.revbuilder-cpicker-presets-group'),
						scroller = group.find('.rs__scrollbar-x-rail');

					colr.remove();
					$('<span id="hello" class="revbuilder-cpicker-color blank"></span>').insertBefore(scroller);

					updatePresets(group);
					cPicker.removeClass('onajax onajaxdelete');

				});

				titl = $.trim(titl.replace(/\W+/g, '_')).replace(/^\_|\_$/g, '').toLowerCase();
				onAjax('delete', titl, tpe, 'revcpicker_onajax_delete', currentInput);

			}

			return false;

		});

		$('.revbuilder-cpicker-save-preset-btn').on('click.revcpicker', function() {

			if(!onAjax) {
				console.log('Ajax callback not defined');
				return;

			}

			var presetGroup,
				$this = $(this),
				input = $this.closest('.revbuilder-cpicker-presets').find('.revbuilder-cpicker-preset-save'),
				titl = input.val(),
				isError;

			if(!titl || !isNaN(titl)) {

				alert($this.attr('data-message'));
				return;

			}

			presetGroup = cPicker.hasClass('color-view') ? 'colors' : 'gradients';
			titl = $.trim(titl.replace(/\W+/g, '_')).replace(/^\_|\_$/g, '').toLowerCase();

			$('#revbuilder-cpicker-' + presetGroup + '-custom').find('.revbuilder-cpicker-color').not('.blank').each(function() {

				var atr = $.trim(getAttribute(this, 'data-title').replace(/\W+/g, '_')).replace(/^\_|\_$/g, '').toLowerCase();
				if(atr === titl) {

					alert($this.attr('data-message'));
					isError = true;
					return false;

				}

			});


			if(isError) return;
			cPicker.addClass('onajax onajaxsave');

			var newColorValue = {},
				color,
				grad;


			if(presetGroup === 'colors') {

				var hex = colorHex.val(),
					opacity = colorOpacity.val();

				if(hex === 'transparent') color = 'transparent';
				else if(opacity === '100%') color = RSColor.sanitizeHex(hex);
				else color = RSColor.processRgba(hex, opacity);

			}
			else {

				grad = gradOutput.style.background;
				buildGradientOutput(false, false, true);
				color = $.extend({}, gradientPreview);

			}

			newColorValue[titl] = color;
			RVS.DOC.off('revcpicker_onajax_save.revcpicker').on('revcpicker_onajax_save.revcpicker', function(evt, err) {
				if(err) {
					cPicker.removeClass('onajax onajaxsave');
					alert($this.attr('data-message'));
					return;
				}

				var pre = $(newPreset(newColorValue, false, ' revbuilder-picker-color-custom', grad)),
					group = $('#revbuilder-cpicker-' + presetGroup + '-custom'),
					scroller = group.find('.rs__scrollbar-x-rail'),
					box = group.find('.revbuilder-cpicker-color.blank');

				if(box.length) {
					box = box.eq(0);
					pre.insertBefore(box);
					box.remove();

				}
				else {
					pre.insertBefore(scroller);
					var len = group.children('.revbuilder-cpicker-color').length,
						rows = Math.max(Math.ceil(len / presetColumns), presetRows),
						max = rows * presetColumns;
					while(len++ < max) $('<span class="revbuilder-cpicker-color blank"></span>').insertBefore(scroller);

				}
				$('#revbuilder-cpicker-preset-' + presetGroup).val('custom').trigger('change');
				updatePresets(group);
				pre.trigger('click');

				$('.revbuilder-cpicker-preset-save').val('');
				cPicker.removeClass('onajax onajaxsave');

			});
			onAjax('save', newColorValue, presetGroup, 'revcpicker_onajax_save', currentInput);


		});

		colorClear.on('click.revcpicker', function() {

			updateSlider(0);
			colorIris.val(RSColor.defaultValue).change();
			colorHex.val('transparent');
			onChange(false, 'transparent');

			clearTimeout(opacityFixer);
			opacityFixer = setTimeout(fixOpacity, 100);

		});

		cPicker.find('input[type="text"]').on('focusin.revcpicker', function() {

			inFocus = this;

		}).on('focusout.revcpicker', function() {

			inFocus = false;

		});

		$('.revbuilder-cpicker-input').on('focusin.revcpicker', function() {

			var $this = $(this);
			$this.data('orig-value', $this.val()).parent().addClass('focus');

		}).on('focusout.revcpicker', function() {

			$(this).parent().removeClass('focus');

		});

		$('.revbuilder-cpicker-rgb').on('focusin.revcpicker', function() {

			$(this).data('prev', this.value);

		}).on('focusout.revcpicker keyup.revcpicker', function() {

			var val = this.value;
			if(isNaN(val) || val < 0 || val > 255) {

				this.value = $(this).data('prev');
				return;

			}

			if(cPicker.hasClass('color-view')) {

				colorHex.val(RSColor.rgbToHex('rgb(' + colorR.value + ',' + colorG.value + ',' + colorB.value + ')')).trigger('focusout');

			}
			else {

				gradHex.val(RSColor.rgbToHex('rgb(' + colorR2.value + ',' + colorG2.value + ',' + colorB2.value + ')')).trigger('focusout');

			}

		});

		$('.revbuilder-cpicker-hex').on('focusout.revcpicker', function() {

			var $this,
				oVal,
				val;

			if(this.id === 'revbuilder-cpicker-color-hex') {

				val = colorHex.val();
				if(val) {

					val = RSColor.sanitizeHex(val);
					if(RSColor.isColor.test(val)) {

						updateColor(val);

					}
					else {

						$this = $(this);
						oVal = $this.data('orig-value');

						if(oVal) {

							val = oVal;
							if(val === 'transparent') {

								colorClear.trigger('click');
								return;

							}

							updateColor(val);

						}
						else {

							colorClear.trigger('click');
							return;

						}

					}

				}
				else {

					colorClear.trigger('click');
					return;

				}

				colorIris.val(val).change();

			}
			else {

				val = gradHex.val() || RSColor.defaultValue;
				val = RSColor.sanitizeHex(val);

				if(!RSColor.isColor.test(val)) {

					$this = $(this);
					oVal = $this.data('orig-value');
					val = oVal ? oVal : RSColor.defaultValue;

				}

				gradHex.val(val);
				gradIris.val(val).change();

			}

		}).on('focusin.revcpicker', focusPatch);

		pointLoc.on('keyup.revcpicker focusout.revcpicker', function(e) {

			if(!curPoint) return;

			var align = curPoint.hasClass('revbuilder-cpicker-point-bottom') ? 'bottom' : 'top',
				locations = align === 'bottom' ? botPoints : topPoints,
				point = pointLoc.val().replace('%', '') || '0',
				evt = e.type,
				dir;

			if(isNaN(point)) point = evt === 'keyup' ? '0' : curPoint.attr('data-location');
			point = Math.max(0, Math.min(100, parseInt(point, 10)));

			dir = point < 50 ? 1 : -1;

			while(locations.indexOf(point) !== -1) point += dir;

			if(evt === 'focusout') {

				pointLoc.val(point + '%');
				curPoint.attr('data-location', point);

			}

			curPoint.css('left', point + '%');
			buildGradientOutput();

		}).on('focusin.revcpicker', focusPatch);

		$('#revbuilder-cpciker-clear-gradient').on('click.revcpicker', function() {

			gradIris.val(RSColor.defaultValue).change();

		});

		angle.on('keyup.revcpicker focusout.revcpicker', onAngleChange).on('focusin.revcpicker', focusPatch);

		directions.on('click.revcpicker', function() {

			var $this = $(this),
				dir = $this.attr('data-direction');

			directions.removeClass('selected');
			$this.addClass('selected');

			if(dir !== 'radial') angle.val(getDegree(dir));
			else angle.val(0);

			onAngleChange(false, dir);

		});

		$('.revbuilder-cpicker-point-delete').on('click.revcpicker', function() {

			if(this.className.search('active') === -1) return;

			var align = curPoint.hasClass('revbuilder-cpicker-point-bottom') ? 'bottom' : 'top',
				len = cPicker.find('.revbuilder-cpicker-point-' + align).length;

			if(len > 2) {

				curPoint.draggable('destroy').remove();
				groupPoints = pointerWrap.children();

				cPicker.trigger('click');
				buildGradientOutput();

			}

			if(len <= maxPoints) {

				if(align === 'bottom') hitBottom.removeClass('full');
				else hitTop.removeClass('full');

			}

		});

		$('.revbuilder-cpicker-preset-save').on('focusin.revcpicker', focusPatch);
		$('.revbuilder-cpicker-opacity-input').on('keyup.revcpicker focusout.revcpicker', function(e) {

			var isColor = this.id.search('grad') === -1,
				$this = isColor ? colorOpacity : gradOpacity,
				opacity = $this.val().replace('%', ''),
				evt = e.type,
				clr;

			if(isNaN(opacity)) opacity = evt === 'keyup' ? '0' : $(this).data('orig-value').replace('%', '');
			opacity = Math.max(0, Math.min(100, opacity));

			if(evt === 'focusout') {

				$this.val(opacity + '%');
				if(!isColor) curPoint.attr('data-opacity', opacity);

			}

			if(isColor) {

				var opaque = parseInt(opacity, 10),
					val = opaque !== 0 ? false : 'transparent';

				onChange(false, val, opacity);
				updateSlider(opacity);

			}
			else {

				var parsed = RSColor.rgbValues(curPoint.attr('data-color'), 3),
					color = gradientPreview.colors[pointIndex];

				opacity = RSColor.sanitizeAlpha(parseInt(opacity, 10) * 0.01);

				color.r = parsed[0];
				color.g = parsed[1];
				color.b = parsed[2];
				color.a = opacity;

				clr = RSColor.rgbaString(color.r, color.g, color.b, opacity);
				curPoint.attr('data-color', clr);
				buildGradientOutput();

				clr = 'rgba(0, 0, 0, ' + opacity + ')';
				curSquare.style.backgroundColor = clr;
				curCorner.style.borderTopColor = clr;

			}

		}).on('focusin.revcpicker', focusPatch);

		$('.revbuilder-cpicker-builder-hit').on('click.revcpicker', function(e) {

			if(!points) calculatePoints();

			var hit = parseInt(((e.pageX - hitTop.offset().left) / hitWidth).toFixed(2) * 100, 10),
				align = this.id.search('bottom') !== -1 ? 'bottom' : 'top',
				locations = align === 'bottom' ? botPoints : topPoints,
				dir = hit < 50 ? 1 : -1;

			while(locations.indexOf(hit) !== -1) hit += dir;
			if(align === 'bottom') {

				if(cPicker.find('.revbuilder-cpicker-point-bottom').length < maxPoints) {

					clonePoint(align, hit);
					selectedColor = false;

				}
				else {

					hitBottom.addClass('full');

				}

			}
			else {

				if(cPicker.find('.revbuilder-cpicker-point-top').length < maxPoints) {

					clonePoint(align, hit);
					selectedColor = false;

				}
				else {

					hitTop.addClass('full');

				}

			}

		});

		reverse.on('click.revcpicker', function() {

			var rev = !reverse.hasClass('checked');

			if(rev) reverse.addClass('checked');
			else reverse.removeClass('checked');

			buildGradientOutput(false, true);

		});

		easing.on('change.revcpicker', function() {

			buildGradientOutput();
			setEasingRow(this.value);

		});

		strength.on('keyup.revcpicker focusout.revcpicker', function(e) {

			var power = strength.val().replace('%', ''),
				evt = e.type;

			if(isNaN(power)) power = evt === 'keyup' ? '0' : strength.data('orig-value').replace('%', '');
			power = Math.max(10, Math.min(1000, power));

			if(evt === 'focusout') strength.val(power + '%');
			buildGradientOutput();


		}).on('focusin.revcpicker', focusPatch);

		var presets = $('.revbuilder-cpicker-preset-selector').on('change.revcpicker', function() {

			var toShow = this.value,
				toHide = toShow === 'core' ? 'custom' : 'core',
				tpe = this.id.search('colors') !== -1 ? 'colors' : 'gradients';

			document.getElementById('revbuilder-cpicker-' + tpe + '-' + toHide).style.display = 'none';
			document.getElementById('revbuilder-cpicker-' + tpe + '-' + toShow).style.display = 'block';
			updatePresets();

		});

		if($.fn.ddTP) {

			presets.ddTP();
			easing.ddTP();

		}

		inited = true;

	}

	function fixOpacity() {

		colorOpacity.val('0%');

	}

	function isRawGradient(val) {

		return typeof val === 'string' && val.trim() !== 'transparent' && val.search(/\[\{/) === -1 && val.search('-gradient') !== -1;

	}

	function addPresets(sets) {

		var settings = $.extend({}, sets),
			core = settings.core || {},
			custom = settings.custom,
			container,
			preset,
			colors,
			len,
			el;

		if(!customAdded || custom) {

			len = 4;
			customAdded = custom;
			custom = customAdded || {'colors': [], 'gradients': []};

		}
		else {

			len = 2;

		}

		if(!core.colors) core.colors = defColors;
		if(!core.gradients) core.gradients = defGradients;

		for(var i = 0; i < len; i++) {

			switch(i) {

				case 0:

					container = 'colors-core';
					colors = core.colors;

				break;

				case 1:

					container = 'gradients-core';
					colors = core.gradients;

				break;

				case 2:

					container = 'colors-custom';
					colors = custom.colors;

				break;

				case 3:

					container = 'gradients-custom';
					colors = custom.gradients;

				break;

			}

			preset = writePresets(container, colors.slice() || []);
			el = $('#' + preset[0]);
			el.find('.revbuilder-cpicker-color').remove();
			el.prepend(preset[1]);

		}
	}

/*
GLOBAL SKIN EXTENSION FOR COLORPICKER
*/
	function rArray(a,removeItem,atr) {
		return jQuery.grep(a, function(value) {return value[atr] != removeItem[atr];});
	};

	function findInObject(c,ar,atr) {
		var r = -1;
		for (var i in ar) if (ar[i][atr]!==undefined && ar[i][atr]===c[atr]) r = i;
		return r;
	}

	function checkCandidates() {
		var found = [];
		for (var c in globalCandidates) {
			found[c] = -1;
			for (var i in  globalColors) {
				if (!globalColors.hasOwnProperty(i) || globalColors[i].ref===undefined || globalColors[i].ref.length===0 || found[c]!==-1) continue;
				found[c] = findInObject(globalCandidates[c],globalColors[i].ref,'r')!==-1 ? i : -1;
			}
		}
		var lastval = -1;
		for (var i in found) {
			lastval = lastval===-1 && found[i] !== -1 ? found[i] : found[i]!==-1 && lastval[i]!==-1 ? -999 : lastval;
		}
		return lastval;
	};

	function formatSkinColor (state) {
	  if (!state.id) {
	    return state.text;
	  }
	  var $state = jQuery('<span><span class="skin_mini_color" style="background:'+globalColors[state.element.value].v+'"></span>' + state.text + '</span>');
	  return $state;
	};

	function setGlobalColor(_) {
		if (globalColors[_]!==undefined) setValue(globalColors[_].v,false);
		for (var c in globalCandidates) {
			for (var i in  globalColors) {
				if (!globalColors.hasOwnProperty(i)) continue;
				if (i!==_ && globalColors[i].ref!==undefined && globalColors[i].ref.length!==0) globalColors[i].ref = rArray(globalColors[i].ref,globalCandidates[c],'r');
				if (i===_) {
					globalColors[i].ref = globalColors[i].ref===undefined ? [] : globalColors[i].ref;
					if (findInObject(globalCandidates[c],globalColors[i].ref,'r')==-1) globalColors[i].ref.push(globalCandidates[c]);
				}
			}
		}
	}


	function checkGradientAvailable(_,mode) {
		var nogradient = mode==="single" || mode==="basic";
		if (!nogradient) for (var i in globalColors[_].ref) if (globalColors[_].ref.hasOwnProperty(i)) nogradient = nogradient===true ? true : RVS.F.noGradient(globalColors[_].ref[i].r);
		if (nogradient) jQuery('#revbuilder-cpicker-gradient-btn').hide(); else jQuery('#revbuilder-cpicker-gradient-btn').show();
	}

	$.rsColorPickerApi = {
		setValue : function(val, fromPreset) {
			setValue(val,fromPreset);	// UPADTE CURRENT SELECTED VALUE
		},

		setGlobalColors : function(_) {
			globalColors = RVS.F.safeExtend(true,{},_);  // SKIN OBJECT STRUCTURE
		},

		setGlobalCandidates : function(_) {
			globalCandidates = _;	// THE ATTRIBUTES WHICH NEED TO BE ADDED/REMOVED FROM THE SKIN
		},

		getGlobalColors : function() {
			return globalColors;
		},

		closeColorPicker : function() {
			onClose();
		},

		showGlobal : function(_,colors,candidates) {

			globalColors = RVS.F.safeExtend(true,{},colors);
			globalCandidates = candidates;
			globalColorMode = _.mode;

			var d = _.currentInput[0].dataset,m,s
				currentEditing = document.getElementById('revbuilder-cpicker-current-edit'),
				editTitle = document.getElementById('revbuilder-cpicker-edit-title'),
				widgetId = currentInput.attr('data-widget-id'),
				html = currentInput.attr('data-appended-html'),
				editing = currentInput.attr('data-editing'),
				em = document.createElement('div');
			em.id =  "revbuilder-cpicker-globals";

			//m = '<div id="revbuilder-cpicker-globals">'
			m ='<div id="revbuilder-cpicker-gactive">'+RVS_LANG.globalcoloractive+'</div>';
			m+='<input id="enadisaGlobalSkin" type="checkbox">';
			m+='<div id="inp_SkinPicker">';
			m+='<div style="margin-left:25px;margin-right:20px;" class="revbuilder-cpicker-input-wrap"><select id="gc_skinPicker">';
			for (var i in globalColors) if (globalColors.hasOwnProperty(i)) {

				m +='<option value="'+i+'" '+(globalColors[i].v.indexOf("gradient")>=0 && (_.mode === 'single' || _.mode === 'basic') ? ' disabled="true" ' : '')+'>'+globalColors[i].alias+'</option>';
			}
			m+= '</select></div>';
			m+='<div id="edit_skin_colors" class="basic_action_button layerinput autosize"><i class="material-icons">color_lens</i>'+RVS_LANG.editskins+'</div></div>';
			em.innerHTML = m;

			document.getElementById('revbuilder-cpicker').appendChild(em);
			s = jQuery(document.getElementById('gc_skinPicker'));

			var cc = checkCandidates();

			if (cc===-1 || cc===-999) {
				globalColorActive = false;
				document.getElementById('inp_SkinPicker').style.display="none";
				jQuery('#enadisaGlobalSkin')[0].checked = false;
			} else {
				globalColorActive = cc;
				s[0].value = cc;
				document.getElementById('inp_SkinPicker').style.display="inline-block";
				jQuery('#enadisaGlobalSkin')[0].checked = true;
				currentEditing.innerHTML = globalColors[cc].alias+" Skin";
			}

			if (RVS!==undefined && RVS.F!==undefined && RVS.F.initOnOff!==undefined) {
				RVS.F.initOnOff(em);
				RVS.F.turnOnOff(em,false)
			}


			s.ddTP("destroy").ddTP({tags:false,
						pre:'<span class="skin_mini_color" style="background:#data#"></span>',
						preDataFunction: function(p) {
							var gC = jQuery.rsColorPickerApi.getGlobalColors();
							return (gC!==undefined && gC[p]!==undefined) ? gC[p].v : undefined
						},
						templateResult:formatSkinColor,

			});


			// TURN GLOBAL SKINS ON/OFF
			jQuery('#enadisaGlobalSkin').on('change',function() {
				forceChange = true;
				if (this.checked===true) {
					document.getElementById('inp_SkinPicker').style.display="inline-block";
					s.trigger('change.select2');
					s.ddTP('change');
				} else {
					//rememberLastGlobalActive = globalColorActive;
					globalColorActive = false;
					setGlobalColor(-1);
					document.getElementById('inp_SkinPicker').style.display="none";
					setValue(openingValue,false);
					currentEditing.innerHTML = editing;
				}
			});

			// SET AN OTHER SKIN
			s.on('change.select2',function() {
				forceChange = true;
				globalColors = RVS.F.safeExtend(true,{},colors);
				if (globalColorActive!==-1 && globalColorActive!==-999 && colors[globalColorActive]!==undefined) setValue(colors[globalColorActive].v,false);
				globalColorActive = this.value;
				setGlobalColor(this.value);
				checkGradientAvailable(this.value,globalColorMode);
				currentEditing.innerHTML = globalColors[this.value].alias+" Skin";
			});
		},

		hideGlobal : function() {
			jQuery('#revbuilder-cpicker-globals').remove();
			globalColors = undefined;
			globalColorActive = undefined;
		}
	};

	$.rsColorPicker = function(settings) {

		origSettings = $.extend(true, origSettings || {}, settings);

		if(settings.mode) defMode = settings.mode;
		if(settings.init) onInit = settings.init;
		if(settings.onAjax) defAjax = settings.onAjax;
		if(settings.onEdit) defEdit = settings.onEdit;
		if(settings.change) defChange = settings.change;
		if(settings.cancel) defCancel = settings.cancel;
		if(settings.title) defTitle = settings.title;
		if(settings.language) customLang = settings.language;
		if(settings.widgetId) defWidgetId = settings.widgetId;
		if(settings.defaultValue) RSColor.defaultValue = settings.defaultValue;
		if(settings.wrapClasses) defaultClasses = settings.wrapClasses;
		if(settings.appendedHtml) appendedHTML = settings.appendedHtml;
		if(settings.onRefresh) defRefresh = settings.defRefresh;

		if(prepped) {
			updatePresets();
			if(settings.language) writeLanguage(settings.language);
		}
	};

	function onStart() {

		if(!bodies) {

			bodies = $('body');
			cPicker = $('<div class="' + 'revbuilder-cpicker-wrap _TPRB_ color-view palette-open">' + markup + '</div>').appendTo(bodies);

		}

		if(origSettings.core) {

			if(origSettings.core.colors) defColors = origSettings.core.colors;
			if(origSettings.core.gradients) defGradients = origSettings.core.gradients;

		}

		addPresets(origSettings);
		writeLanguage(origSettings.language || lang);
		prepped = true;




	}

	var ColorPicker = {

		refresh: function() {

			var $this = $(this),
				ar = [$this];

			if($this.hasClass('revbuilder-cpicker-component')) {

				var settings = $this.data('revcp') || {},
					val = $this.val() || settings.defaultValue || RSColor.defaultValue,
					refresh = settings.onRefresh || defRefresh,
					colorValue = RSColor.process(val);

				val = colorValue[0];
				colorValue = colorValue[1] !== 'rgba' || !RSColor.transparentRgba(val, true) ? val : '';

				if(val !== 'transparent') {
					$this.data('tpcp')[0].style.background = colorValue;
					if(previewColor) previewColor.css('background', colorValue);
				}
				else {
					$this.data('tpcp').css('background', '');
					if(previewColor) previewColor.css('background', '');
				}

				$this.attr('data-color', val).data('hex', val);

				ar[1] = val;
				if(refresh) refresh(ar[0], val);

			}
			else {

				ar[1] = false;
				ar[2] = 'Color Picker not inited for this input field yet';

			}

			RVS.DOC.trigger('revcolorpickerrefresh', ar);

		},

		destroy: function() {

			var $this = $(this).removeData();
			$this.closest('.revbuilder-cpicker-master-wrap').removeData().remove();

		}

	};

	$.fn.rsColorPicker = function(settings) {

		if(settings && typeof settings === 'string') return this.each(ColorPicker[settings]);

		return this.each(function() {

			var $this = $(this);

			if(this.className.indexOf('revbuilder-cpicker-component')>=0) {
				$this.rsColorPicker('refresh');
				return;
			}


			var wrap = document.createElement('div'),
				box = document.createElement('span'),
				btn = document.createElement('span');

			box.className = 'revbuilder-colorbox';
			btn.className = 'revbuilder-colorbtn';
			wrap.className ='revbuilder-colorpicker';
			wrap.appendChild(box);
			wrap.appendChild(btn);


			var cls = this.dataset.wrapClasses || defaultClasses, // $this.attr('data-wrap-classes'),
				wrapper = this.dataset.wrapper, // $this.attr('data-wrapper'),
				ids = this.dataset.wrapId, //$this.attr('data-wrap-id'),
				txt = this.dataset.title, //$this.attr('data-title'),
				val = this.value, //$this.val(),
				classextension = $this.data('visible') == true ? 'visible-cp-input' : '',
				colorValue,
				defValue;



			btn.innerText = txt || defTitle || langColor || lang.color;


			if(cls!==undefined) wrap.classList.add(cls);
			if(ids) wrap.id = ids;
			if(settings && $.isPlainObject(settings)) {

				if(!wrapper) wrapper = settings.wrapper;
				if(!cls) cls = settings.wrapClasses;
				if(!ids) ids = settings.wrapId;
				if(!txt) txt = settings.title;

				defValue = settings.defaultValue;
				var sets = $this.data('revcp');

				if(sets) settings = $.extend({}, sets, settings);
				$this.data('revcp', settings);

			}

			if(!val) {
				val = defValue || RSColor.defaultValue;
				this.value = val;
			}

			colorValue = RSColor.process(val);
			val = colorValue[0];

			colorValue = colorValue[1] !== 'rgba' || !RSColor.transparentRgba(val, true) ? val : '';
			if(colorValue !== 'transparent') box.style.background = colorValue;

			this.dataset.color = val;
			this.classList.add('revbuilder-cpicker-component');

			$this.data('tpcp', jQuery(box));

			wrap = this.parentNode.insertBefore(wrap,this);
			wrap.appendChild(this);

			wrap = jQuery(wrap);
			wrap.data('revcolorinput',$this);

			if(!wrapper) {
				wrap.addClass('revbuilder-cpicker-master-wrap').addClass(classextension);
			}
			else {
				wrapper = $(wrapper).addClass('revbuilder-cpicker-master-wrap').addClass(classextension);
				wrap.wrap(wrapper);
			}

			var initCallback = settings ? settings.init || onInit : false,
				options = origSettings ? $.extend(true, origSettings, settings || {}) : settings || false;

			if(initCallback) initCallback(wrap, $this, val, options);
			RVS.DOC.trigger('revcolorpickerinit', [wrap, $this, val, options]);

		});
	};

	$(function() {

		$('body').on('click.revcpicker', '.revbuilder-colorpicker', function() {

			if(!inited) init();

			writeVars();
			currentInput = $(this).data('revcolorinput');

			var currentEditing = document.getElementById('revbuilder-cpicker-current-edit'),
				editTitle = document.getElementById('revbuilder-cpicker-edit-title'),
				widgetId = currentInput.attr('data-widget-id'),
				html = currentInput.attr('data-appended-html'),
				editing = currentInput.attr('data-editing'),
				data = currentInput.attr('data-colors'),
				mode = currentInput.attr('data-mode'),
				settings = currentInput.data('revcp'),
				lang = currentInput.attr('data-lang'),
				settingsGradients,
				customGradients,
				settingsColors,
				dataGradients,
				customColors,
				dataColors,
				presets,
				change,
				cancel,
				value,
				edit,
				ajax,
				val,
				tpe;

			if(data) {

				data = JSON.parse(data.replace(/\&/g, '"'));
				if(data.colors) dataColors = data.colors;
				if(data.gradients) dataGradients = data.gradients;

			}

			if(settings) {

				var colorSets = settings.colors;
				if(colorSets) {

					if(colorSets.core) {

						settingsColors = colorSets.core.colors;
						settingsGradients = colorSets.core.gradients;

					}

					if(colorSets.custom) {

						customColors = colorSets.custom.colors;
						customGradients = colorSets.custom.gradients;

					}

				}

				edit = settings.onEdit;
				ajax = settings.onAjax;
				change = settings.change;
				cancel = settings.cancel;

				if(!lang) lang = settings.lang;
				if(!mode) mode = settings.mode;
				if(!html) html = settings.appendedHtml;
				if(!editing) editing = settings.editing;
				if(!widgetId) widgetId = settings.widgetId;

			}

			if(settingsGradients || settingsColors || customGradients || customColors || dataGradients || dataColors) {

				presets = {};
				if(settingsGradients || settingsColors || dataGradients || dataColors) {

					presets.core = {

						colors: dataColors || settingsColors || defColors,
						gradients: dataGradients || settingsGradients || defGradients

					};

				}

				if(customGradients || customColors) {

					presets.custom = {

						colors: customColors || defColors,
						gradients: customGradients || defGradients

					};

				}

				addPresets(presets);

			}

			if(!widgetId) widgetId = defWidgetId;
			if(widgetId) cPicker[0].id = widgetId;

			if(!html) html = appendedHTML;
			if(html) appended = $(html).appendTo(mainContainer);

			if(lang) writeLanguage(lang);
			if(!mode) mode = defMode;

			if(!editing) {

				editing = '';
				editTitle.style.visibility = 'hidden';

			}
			else {

				editTitle.style.visibility = 'visible';

			}

			currentEditing.innerHTML = editing;

			if(mode === 'single' || mode === 'basic') {

				isFull = false;
				gradBtn.hide();
				colorBtn.show();
				if(mode === 'basic') cPicker.addClass('is-basic');

			}
			else {

				isFull = true;
				gradBtn.show();
				colorBtn.show();

			}

			val = currentInput.val() || RSColor.defaultValue;

			if(isRawGradient(val)) val = RSColor.reverseGradient(val);
			if(val.split('||').length > 1) {

				val = RSColor.joinToRgba(val);
				currentInput.val(val);

			}

			value = setValue(val);
			openingValue = value[0];

			storedColor = openingValue;
			storedValue = currentInput.val();

			onEdit = edit || defEdit;
			onAjax = ajax || defAjax;

			onCancel = cancel || defCancel;
			changeCallback = change || defChange;

			if(value[1] !== 'gradient') colorBtn.data('state', openingValue);
			else gradBtn.data('state', openingValue);

			bodies.addClass('revbuilder-colorpicker-open');
			currentColor = currentInput.data('tpcp');
			cPicker.data('revcpickerinput', currentInput).addClass('active').show();

			updatePresets();
			openingColor = currentInput.attr('data-color');

			var prevColor = RSColor.process(val);

			if(prevColor[1] === 'rgba') {
				prevColor = RSColor.rgbValues(prevColor[0], 4);
				if(prevColor[3] == 0) previewColor.css('background', '');
				else previewColor.css('background', openingColor);
			}
			else if(prevColor[1] === 'transparent') {
				previewColor.css('background', '');
			}
			else {
				previewColor.css('background', openingColor);
			}

			selectedHex = currentInput.data('hex');
			$('.revbuilder-cpicker-color').not('.blank').each(checkPreset);

			mainContainer.css({

				marginLeft: -(mainContainer.width() * 0.5),
				marginTop: -(mainContainer.height() * 0.5)

			});

			// FIX FOR WP BAD BEAHVIOUR ON IRIS PICKER
			var isv = jQuery('.iris-square-value');
			if (isv.data('badlistener')!==true) {
				isv.click(function() {
					this.href = '';
					return false;
				}).data('badlistener',true);
			}

			// CALL BACK
			RVS.DOC.trigger("RSCOLOR.OPEN", {openingValue:value[0], mode:mode, currentInput:currentInput});

		});

	});

	function colorPickerOnLoaded() {
		if(typeof RVS.LIB.COLOR_PRESETS === 'undefined' || !RVS.LIB.COLOR_PRESETS) {
			RVS.LIB.COLOR_PRESETS = {colors: [], gradients: []};
		}

		if(!RVS.LIB.COLOR_PRESETS.colors) RVS.LIB.COLOR_PRESETS.colors = [];
		if(!RVS.LIB.COLOR_PRESETS.gradients) RVS.LIB.COLOR_PRESETS.gradients = [];

		$.rsColorPicker({

			custom: RVS.LIB.COLOR_PRESETS,
			onAjax: function(action, data, colorType, eventToTrigger, currentInputElement) {

				var do_save = false,
					colors = RVS.LIB.COLOR_PRESETS[colorType];

				switch(action) {
					case 'save':
						colors[colors.length] = data;
						do_save = true;
					break;

					case 'delete':
						var len = colors.length;

						dance:
						for(var i = 0; i < len; i++) {
							for(var prop in colors[i]) {
								if(prop === data) {

									colors.splice(i, 1);
									do_save = true;
									break dance;

								}
							}
						}

					break;
				}

				if(do_save && typeof RVS !== 'undefined' && typeof RVS.F !== 'undefined'){
					RVS.F.ajaxRequest('save_color_preset', {presets: RVS.F.safeExtend(true,{},RVS.LIB.COLOR_PRESETS)}, function(response){
						var e = response.error || false;
                        try{if ($.esgColorPicker!==undefined) $.esgColorPicker({custom:RVS.LIB.COLOR_PRESETS});} catch(e) {}
						RVS.DOC.trigger(eventToTrigger, [e]);
					});
				}
			}

		});
    }

    // Color Picker Presets
    var colorPicker_once = false
    if (document.readyState === "loading")
        document.addEventListener('readystatechange',function(){
            if ((document.readyState === "interactive" || document.readyState === "complete") && !colorPicker_once) {
                colorPicker_once = true;
                colorPickerOnLoaded();
            }
        });
    else {
        colorPicker_once = true;
        colorPickerOnLoaded();
    }
    
})(jQuery !== 'undefined' ? jQuery : false);
