define([
	'jquery',
	'mage/translate',
	'Magento_Ui/js/modal/modal',
	'mage/storage',
	'Magento_Customer/js/customer-data',
	'mage/cookies',
	'mage/apply/main',
	'Magento_Catalog/js/price-utils',
	'Noesis_TailorPro/js/price-tailor',
	'Noesis_MeasurementIntegration/js/autocomplete-off-patch'
], function (
	$, $t, modal, storage, customerData, cookies, mageApply, priceUtils, _priceTailor, ensureAutocompleteOffPatch
) {
	'use strict';

	var patchAutocompleteOff = ensureAutocompleteOffPatch;

	ensureAutocompleteOffPatch();

	var DEFAULT_PRECISION = 2;
	var FRACTION_REGEX = /^\s*(\d+)?(?:\s+)?(\d+\/\d+)?\s*$/;

	return function initMeasurementIntegration(config, element) {
		var $root = $(element);

		if (!$root.length) {
			return;
		}

		var state = {
			config: $.extend(true, {
				measurement: {
					fields: [],
					fieldsByType: {},
					instructions: {},
					units: [],
					types: [],
					mode: 'body'
				},
				customer: {
					isLoggedIn: false
				},
				login: {},
				urls: {},
				routes: {},
				price: {
					format: {},
					base: 0
				}
			}, config || {}),
			$root: $root,
			$placeholder: null,
			$productForm: null,
			selection: null,
			loading: false,
			postLoginAction: null,
			profileCache: null,
			profileRequest: null,
			fieldDefinitionCache: {},
			profileDetailsCache: {},
			formValidationBound: false,
			nativeFormSubmit: null,
			entry: {
				ready: false,
				fields: {}
			}
		};

		state.config.measurement.mode = normalizeMode(state.config.measurement.mode);

		cacheDom();
		prepareData();
		bindEvents();
		setLoginState();
		applyPreconfiguredSelection();
		bindTailorIntegration();
		bindFormValidation();
		refreshAddToCartEnabled();

		function cacheDom() {
			state.$loginGate = state.$root.find('[data-role="login-gate"]');
			state.$actions = state.$root.find('[data-role="actions"]');
			state.$status = state.$root.find('[data-role="status"]');
			state.$entryPanel = state.$root.find('[data-role="entry-panel"]');
			state.$summary = state.$root.find('[data-role="summary"]');
			state.$summaryTitle = state.$summary.find('[data-role="summary-title"]');
			state.$summaryMeta = state.$summary.find('[data-role="summary-meta"]');
			state.$summaryList = state.$summary.find('[data-role="summary-breakdown"]');
			state.$summaryMeasurements = state.$summary.find('[data-role="summary-measurements"]');
			state.$summaryTotal = state.$summary.find('[data-role="summary-total"]');
			state.$modalContainer = state.$root.find('[data-role="modal-container"]');
			state.$profileId = state.$root.find('[data-role="profile-id"]');
			state.$payloadInput = state.$root.find('[data-role="profile-payload"]');
			state.$priceInput = state.$root.find('[data-role="profile-price"]');
			state.$enterButton = state.$actions.find('[data-action="enter-new"]');
			state.$loadButton = state.$actions.find('[data-action="load-saved"]');
			state.$addToCartButtons = findAddToCartButtons();
			state.$placeholder = findPlaceholder();
			state.enterDefaultLabel = state.$enterButton.data('label-default') || state.$enterButton.text();
			state.enterEditLabel = state.$enterButton.data('label-edit') || $t('Edit Measurements');
		}

		function prepareData() {
			state.measurementTypes = Array.isArray(state.config.measurement.types)
				? state.config.measurement.types.slice() : [];
			state.typeLabels = {};
			state.measurementTypes.forEach(function (type) {
				var code = normalizeMode(type && (type.code || type.value));
				if (code) {
					state.typeLabels[code] = type.label || code;
				}
			});

			state.measurementUnits = Array.isArray(state.config.measurement.units)
				? state.config.measurement.units.slice() : [];
			state.unitLabels = {};
			state.measurementUnits.forEach(function (unit) {
				var code = unit && (unit.code || unit.value);
				if (code) {
					state.unitLabels[String(code)] = unit.label || code;
				}
			});

			state.fieldsByType = $.extend(true, {}, state.config.measurement.fieldsByType || {});
			state.fields = prepareFields(state.config.measurement.fields || []);
			if (!state.fields.length) {
				state.fields = [];
			}

			if (!Object.keys(state.fieldsByType).length && state.fields.length) {
				state.fieldsByType[state.config.measurement.mode] = state.fields;
			}

			state.fieldMap = buildFieldMap(state.fieldsByType, state.fields);
			state.activeMode = state.config.measurement.mode;
			state.unitValue = resolveDefaultUnit();
		}

		function bindEvents() {
			state.$root.on('click', '[data-action="login"]', function () {
				ensureAuthenticated(function () {
					/* no-op */
				}, true);
			});

			state.$enterButton.on('click', function () {
				handleEnterNew();
			});

			state.$loadButton.on('click', function () {
				handleLoadSaved();
			});

			state.$entryPanel.on('submit', 'form', function (event) {
				event.preventDefault();
				applyEntry();
			});

			state.$entryPanel.on('click', '[data-action="apply-entry"]', function (event) {
				event.preventDefault();
				applyEntry();
			});

			state.$entryPanel.on('click', '[data-action="close-entry"]', function (event) {
				event.preventDefault();
				hideEntryPanel();
			});

			state.$entryPanel.on('click', '[data-action="reset-entry"]', function (event) {
				event.preventDefault();
				resetEntryForm();
			});

			state.$entryPanel.on('change', '[data-role="unit-option"]', function () {
				state.unitValue = $(this).val();
				updateUnitIndicators();
			});

			state.$entryPanel.on('change', 'input[name="measurement_type"]', function () {
				switchMeasurementMode($(this).val());
			});

			state.$entryPanel.on('blur', '[data-field-type="numeric"]', function () {
				normalizeNumericInput($(this));
			});

			state.$entryPanel.on('input change', '[data-field-type="numeric"],[data-field-type="textarea"]', function () {
				clearFieldError($(this).data('field-id'));
				updateFieldPricingNote($(this).data('field-id'), '');
				if ($(this).data('field-type') === 'numeric') {
					updateFieldConversions();
				}
			});

			state.$entryPanel.on('focus', '.measurement-step__field-control', function () {
				showFieldInstructions($(this).data('field-id'));
			});
		}

		function setLoginState() {
			var requireLogin = !!state.config.requireLogin;
			var loggedIn = !!(state.config.customer && state.config.customer.isLoggedIn);

			if (requireLogin && !loggedIn) {
				state.$loginGate.show();
				state.$actions.hide();
			} else {
				state.$loginGate.hide();
				state.$actions.show();
			}
		}

		function applyPreconfiguredSelection() {
			var raw = (state.$payloadInput.val() || '').trim();
			if (!raw) {
				return;
			}

			var payload;

			try {
				payload = JSON.parse(raw);
			} catch (error) {
				payload = null;
			}

			if (!payload || typeof payload !== 'object') {
				return;
			}

			var selection = selectionFromPayload(payload);
			if (!selection) {
				return;
			}

			applySelection(selection, { silent: true });
		}

		function handleEnterNew() {
			ensureEntryPanel();

			if (state.selection) {
				populateEntry(state.selection);
			} else {
				resetEntryForm();
			}

			showEntryPanel();
		}

		function handleLoadSaved() {
			ensureAuthenticated(function () {
				setLoadButtonState(true);
				showProfilesLoading();
				var request = toPromise(storage.get(state.config.urls.list, false));
				request.then(function (response) {
					if (!response || !response.success) {
						throw new Error(response && response.message ? response.message : $t('Unable to load saved measurements.'));
					}

					var profiles = normalizeProfiles(response.profiles || []);
					state.profileCache = profiles;
					renderProfileModal(profiles);
					setLoadButtonState(false);
				}).catch(function (error) {
					showProfileModalError(error && error.message ? error.message : $t('Unable to load saved measurements.'));
					setLoadButtonState(false);
				});
			}, true);
		}

		function ensureAuthenticated(callback, forceModal) {
			var loggedIn = !!(state.config.customer && state.config.customer.isLoggedIn);
			var requireLogin = !!(forceModal || state.config.requireLogin);

			if (!requireLogin || loggedIn) {
				if (typeof callback === 'function') {
					callback();
				}
				return;
			}

			setLoadButtonState(false);
			state.postLoginAction = typeof callback === 'function' ? callback : null;
			openAuthModal(forceModal);
		}

		function openAuthModal(forceModal) {
			if (state.authModal && state.authModal.modal) {
				state.authModal.modal('openModal');
				return;
			}

			var $wrapper = $('<div/>', { class: 'measurement-step__auth-modal' });
			var description = $('<div/>', {
				class: 'measurement-step__auth-description'
			}).text($t('Sign in or create an account to save and reuse your measurements.'));

			var columns = $('<div/>', { class: 'measurement-step__auth-columns' });
			columns.append(buildAuthForm('login'));
			columns.append(buildAuthForm('register'));

			$wrapper.append(description).append(columns);
			$('body').append($wrapper);

			modal({
				title: $t('Account Access'),
				modalClass: 'measurement-step__auth-wrapper',
				zIndex: 1200,
				buttons: [{
					text: $t('Close'),
					class: 'action secondary',
					click: function () {
						this.closeModal();
					}
				}]
			}, $wrapper);

			state.authModal = $wrapper;
			state.authModal.modal('openModal');
		}

		function buildAuthForm(mode) {
			var $form = $('<form/>', {
				class: 'measurement-step__auth-form',
				'data-mode': mode
			});

			var title = mode === 'login' ? $t('Sign In') : $t('Create Account');
			$form.append($('<h3/>').text(title));

			if (mode === 'register') {
				$form.append(buildAuthField('firstname', $t('First Name')));
				$form.append(buildAuthField('lastname', $t('Last Name')));
			}

			$form.append(buildAuthField(mode === 'login' ? 'username' : 'email', $t('Email Address'), 'email'));
			$form.append(buildAuthField('password', $t('Password'), 'password'));

			var $actions = $('<div/>', { class: 'measurement-step__auth-actions' });
			var $submit = $('<button/>', {
				type: 'submit',
				class: 'action primary'
			}).append($('<span/>').text(title));
			$actions.append($submit);

			$form.append($actions);
			$form.append($('<div/>', {
				class: 'measurement-step__auth-message',
				'data-role': mode + '-message'
			}));

			$form.on('submit', function (event) {
				event.preventDefault();
				processAuth(mode, $form);
			});

			return $form;
		}

		function buildAuthField(name, label, type) {
			return $('<div/>', { class: 'field' }).append(
				$('<label/>').text(label),
				$('<input/>', {
					type: type || 'text',
					name: name,
					class: 'input-text',
					required: true
				})
			);
		}

		function processAuth(mode, $form) {
			var message = $form.find('[data-role="' + mode + '-message"]');
			message.empty();

			var payload = {
				form_key: getFormKey()
			};

			if (mode === 'login') {
				payload.username = ($form.find('[name="username"]').val() || '').trim();
				payload.password = $form.find('[name="password"]').val() || '';
			} else {
				payload.firstname = ($form.find('[name="firstname"]').val() || '').trim();
				payload.lastname = ($form.find('[name="lastname"]').val() || '').trim();
				payload.email = ($form.find('[name="email"]').val() || '').trim();
				payload.password = $form.find('[name="password"]').val() || '';
			}

			toggleFormState($form, true);

			var url = mode === 'login' ? state.config.login.loginUrl : state.config.login.registerUrl;

			var request = toPromise(storage.post(url, JSON.stringify(payload), false, 'application/json'));
			request.then(function (response) {
				if (!response || !response.success) {
					throw new Error(response && response.message ? response.message : $t('Unable to complete the request.'));
				}

				onAuthenticationSuccess(response);
				toggleFormState($form, false);
			}).catch(function (error) {
				message.text(error && error.message ? error.message : $t('Unable to complete the request.'));
				toggleFormState($form, false);
			});
		}

		function toggleFormState($form, disabled) {
			$form.find('input, button').prop('disabled', !!disabled);
		}

		function onAuthenticationSuccess(response) {
			state.config.customer.isLoggedIn = true;
			customerData.invalidate(['customer']);
			customerData.reload(['customer'], true);

			if (response && response.formKey) {
				updateFormKey(response.formKey);
			}

			if (state.authModal) {
				state.authModal.modal('closeModal');
			}

			setLoginState();
			setStatus('success', $t('You are now signed in.'));

			var callback = state.postLoginAction;
			state.postLoginAction = null;
			if (typeof callback === 'function') {
				callback();
			}
		}

		function ensureEntryPanel() {
			if (state.entry.ready) {
				return;
			}

			buildEntryPanel();
			initializeEntryModal();
			state.entry.ready = true;
		}

		function buildEntryPanel() {
			var $panel = state.$entryPanel.empty();
			var $wrapper = $('<div/>', { class: 'measurement-profile-modal measurement-profile-modal--product' });
			var $content = $('<div/>', { class: 'measurement-profile-modal__content' });
			var $form = $('<form/>', {
				class: 'measurement-profile-form measurement-profile-form--product',
				novalidate: 'novalidate'
			});

			state.entry.$form = $form;

			state.entry.$error = $('<div/>', {
				class: 'message message-error measurement-profile-form__error',
				'data-role': 'entry-error',
				style: 'display:none'
			});
			$form.append(state.entry.$error);

			var $optionsFieldset = $('<fieldset/>', { class: 'fieldset' });
			var $optionsGrid = $('<div/>', { class: 'measurement-profile-form__options-grid' });

			var $typeOption = $('<div/>', { class: 'field required measurement-profile-form__option' });
			$typeOption.append($('<label/>', { class: 'label' }).append($('<span/>').text($t('Measurement Type'))));
			state.entry.$typeContainer = $('<div/>', {
				class: 'options-list nested measurement-profile-form__choice-list js-measurement-type-options',
				'data-role': 'measurement-type-options'
			});
			$typeOption.append($('<div/>', { class: 'control' }).append(state.entry.$typeContainer));
			$optionsGrid.append($typeOption);

			var $unitOption = $('<div/>', { class: 'field required measurement-profile-form__option' });
			$unitOption.append($('<label/>', { class: 'label' }).append($('<span/>').text($t('Measurement Unit'))));
			state.entry.$unitContainer = $('<div/>', {
				class: 'options-list nested measurement-profile-form__choice-list js-measurement-unit-group',
				'data-role': 'measurement-unit-options'
			});
			$unitOption.append($('<div/>', { class: 'control' }).append(state.entry.$unitContainer));
			$optionsGrid.append($unitOption);

			$optionsFieldset.append($optionsGrid);
			$form.append($optionsFieldset);

			state.entry.$generalInstructionWrapper = $('<div/>', {
				class: 'measurement-profile-form__instructions js-form-general-wrapper',
				style: 'display:none'
			});
			state.entry.$generalInstructionContent = $('<div/>', {
				class: 'instruction-content js-form-general'
			});
			state.entry.$generalInstructionWrapper
				.append($('<h3/>').text($t('How to Measure')))
				.append(state.entry.$generalInstructionContent);
			$form.append(state.entry.$generalInstructionWrapper);

			state.entry.$modeInstructionWrapper = $('<div/>', {
				class: 'measurement-profile-form__instructions js-form-measurement-wrapper',
				style: 'display:none'
			});
			state.entry.$modeInstructionTitle = $('<h4/>', { class: 'js-form-measurement-heading' });
			state.entry.$modeInstructionContent = $('<div/>', { class: 'instruction-content js-form-measurement' });
			state.entry.$modeInstructionWrapper
				.append(state.entry.$modeInstructionTitle)
				.append(state.entry.$modeInstructionContent);
			$form.append(state.entry.$modeInstructionWrapper);

			var $layout = $('<div/>', { class: 'measurement-profile-form__grid' });
			var $fieldsContainer = $('<div/>', { class: 'measurement-profile-form__fields-container' });
			state.entry.$fieldsWrapper = $('<div/>', {
				class: 'fieldset measurement-profile-form__fields js-measurement-field-wrapper',
				'data-role': 'fields-wrapper'
			});
			$fieldsContainer.append(state.entry.$fieldsWrapper);
			$layout.append($fieldsContainer);

			var $sidebar = $('<aside/>', {
				class: 'measurement-profile-form__sidebar',
				'data-role': 'field-instructions-container'
			});
			state.entry.$instructionPlaceholderHeading = $('<h4/>', {
				class: 'js-field-instructions-placeholder-heading',
				style: 'display:none'
			});
			state.entry.$instructionPlaceholderContent = $('<div/>', {
				class: 'instruction-content js-field-instructions-placeholder-content'
			}).append($('<p/>').text($t('Select a measurement field to see detailed guidance.')));
			state.entry.$instructionPlaceholder = $('<div/>', {
				class: 'measurement-profile-form__sidebar-placeholder js-field-instructions-placeholder'
			})
				.append(state.entry.$instructionPlaceholderHeading)
				.append(state.entry.$instructionPlaceholderContent);

			state.entry.$fieldInstructionWrapper = $('<div/>', {
				class: 'measurement-profile-form__field-instructions js-field-instructions',
				style: 'display:none'
			});
			state.entry.$fieldInstructionTitle = $('<h4/>', {
				class: 'js-field-instructions-heading'
			});
			state.entry.$fieldInstructionContent = $('<div/>', {
				class: 'instruction-content js-field-instructions-content'
			});
			state.entry.$fieldInstructionWrapper
				.append(state.entry.$fieldInstructionTitle)
				.append(state.entry.$fieldInstructionContent);

			$sidebar.append(state.entry.$instructionPlaceholder).append(state.entry.$fieldInstructionWrapper);
			$layout.append($sidebar);

			$form.append($layout);

			var $actions = $('<div/>', { class: 'modal-actions measurement-profile-form__actions' });
			$actions.append($('<button/>', {
				type: 'button',
				class: 'action secondary',
				'data-action': 'close-entry'
			}).append($('<span/>').text($t('Cancel'))));
			$actions.append($('<button/>', {
				type: 'button',
				class: 'action secondary',
				'data-action': 'reset-entry'
			}).append($('<span/>').text($t('Clear'))));
			$actions.append($('<button/>', {
				type: 'submit',
				class: 'action primary',
				'data-action': 'apply-entry'
			}).append($('<span/>').text($t('Apply measurements'))));
			$form.append($actions);

			$content.append($form);
			$wrapper.append($content);
			$panel.append($wrapper);

			populateTypeOptions();
			populateUnitOptions();
			renderFieldList();
			renderInstructions();
			updateUnitIndicators();
		}

		function initializeEntryModal() {
			modal({
				type: 'slide',
				title: $t('Enter Measurements'),
				modalClass: 'measurement-profile-modal measurement-profile-modal--product',
				zIndex: 1200,
				buttons: []
			}, state.$entryPanel);
		}

		function showEntryPanel() {
			if (typeof state.$entryPanel.modal === 'function') {
				state.$entryPanel.modal('openModal');
			}
		}

		function hideEntryPanel() {
			if (typeof state.$entryPanel.modal === 'function') {
				state.$entryPanel.modal('closeModal');
			}
		}

		function resetEntryForm() {
			Object.keys(state.entry.fields).forEach(function (key) {
				var entry = state.entry.fields[key];
				entry.$input.val('');
				updateFieldPricingNote(key, '');
				clearFieldError(key);
			});

			state.entry.$error.hide().text('');
			resetFieldInstructions();
		}

		function populateEntry(selection) {
			resetEntryForm();

			if (!selection) {
				syncTypeOptions(state.activeMode);
				syncUnitOptions(state.unitValue);
				return;
			}

			var mode = selection.type || state.activeMode;
			switchMeasurementMode(mode);

			var unit = selection.unit || state.unitValue;
			state.unitValue = unit;
			syncUnitOptions(unit);

			var measurements = selection.measurements || {};
			Object.keys(state.entry.fields).forEach(function (fieldId) {
				var entry = state.entry.fields[fieldId];
				var value = resolveMeasurementValue(measurements, fieldId);

				if (value === null || value === undefined) {
					entry.$input.val('');
					return;
				}

				if (entry.field.field_type === 'numeric') {
					if (typeof value === 'number') {
						entry.$input.val(formatDecimal(value));
					} else {
						var parsed = parseMeasurement(value);
						entry.$input.val(parsed !== null ? formatDecimal(parsed) : value);
					}
				} else {
					entry.$input.val(value);
				}
			});

			syncUnitOptions(state.unitValue);
			updateUnitIndicators();
		}

		function resolveMeasurementValue(map, fieldId) {
			if (!map) {
				return null;
			}
			if (Object.prototype.hasOwnProperty.call(map, fieldId)) {
				return map[fieldId];
			}
			var numericId = parseInt(fieldId, 10);
			if (!isNaN(numericId) && Object.prototype.hasOwnProperty.call(map, numericId)) {
				return map[numericId];
			}
			var stringId = String(fieldId);
			if (Object.prototype.hasOwnProperty.call(map, stringId)) {
				return map[stringId];
			}
			return null;
		}

		function applyEntry() {
			var collected = collectMeasurements();
			if (!collected.valid) {
				state.entry.$error.text($t('Please fix the highlighted fields.')).show();
				return;
			}

			state.entry.$error.hide().text('');
			var selection = buildSelection({
				source: 'manual',
				measurements: collected.measurements
			});

			applySelection(selection);
			hideEntryPanel();
		}

		function collectMeasurements() {
			var measurements = {};
			var valid = true;

			Object.keys(state.entry.fields).forEach(function (key) {
				var entry = state.entry.fields[key];
				var raw = (entry.$input.val() || "").trim();

				if (!raw) {
					if (entry.field.is_required) {
						showFieldError(key, $t('This field is required.'));
						valid = false;
					}
					return;
				}

				if (entry.field.field_type === 'numeric') {
					var parsed = parseMeasurement(raw);
					if (parsed === null) {
						showFieldError(key, $t('Enter a valid number.'));
						valid = false;
						return;
					}
					measurements[key] = parsed;
				} else {
					measurements[key] = raw;
				}
			});

			return {
				valid: valid,
				measurements: measurements
			};
		}

		function applySelection(selection, options) {
			options = options || {};
			if (!selection) {
				return;
			}

			selection.type = normalizeMode(selection.type || state.activeMode);
			selection.typeLabel = selection.typeLabel || getMeasurementTypeLabel(selection.type);
			selection.unit = selection.unit || state.unitValue;
			selection.unitLabel = selection.unitLabel || getUnitLabel(selection.unit);
			selection.measurements = selection.measurements || {};

			var recalculatedAdjustments = calculateAdjustments(selection.measurements, selection.type);
			selection.adjustments = recalculatedAdjustments;
			selection.total = round(recalculatedAdjustments.reduce(function (carry, item) {
				return carry + (Number(item.amount) || 0);
			}, 0), 2);

			selection.fieldOrder = selection.fieldOrder && selection.fieldOrder.length
				? selection.fieldOrder : buildFieldOrder({ type: selection.type });

			if (!selection.summaryRows || !selection.summaryRows.length) {
				selection.summaryRows = buildSummaryRows({
					measurements: selection.measurements,
					unit: selection.unit,
					unitLabel: selection.unitLabel,
					type: selection.type,
					fieldOrder: selection.fieldOrder
				});
			}

			selection.summaryTitle = selection.summaryTitle || defaultSummaryTitle(selection);
			selection.summaryMeta = selection.summaryMeta || buildSummaryMeta(selection);

			state.selection = selection;
			state.activeMode = selection.type || state.activeMode;
			state.unitValue = selection.unit || state.unitValue;

			renderSummary(selection);
			updateHiddenFields(selection);
			setStatus();
			togglePlaceholder(true);
			updateEnterButton(true);
			updatePriceBox(selection.total || 0);
			pushSummaryToTailor(buildTailorPayload(selection));
			setAddToCartEnabled(true);

			if (!options.silent) {
				state.$summary.show();
			}
		}

		function renderSummary(selection) {
			var rows = Array.isArray(selection.summaryRows) && selection.summaryRows.length
				? selection.summaryRows : buildSummaryRows(selection);
			var adjustments = Array.isArray(selection.adjustments) ? selection.adjustments : [];
			var total = Number(selection.total);
			if (!isFinite(total)) {
				total = 0;
			}

			state.$summaryTitle.text(selection.summaryTitle || defaultSummaryTitle(selection));
			state.$summaryMeta.text(selection.summaryMeta || buildSummaryMeta(selection));

			renderMeasurementRows(rows);

			state.$summaryList.empty();
			if (!adjustments.length) {
				state.$summaryList.append($('<em/>').text($t('No additional charges.')));
			} else {
				var $list = $('<ul/>', { class: 'measurement-step__summary-list' });
				adjustments.forEach(function (item) {
					$list.append($('<li/>').text(item.label + ' — ' + formatPrice(item.amount)));
				});
				state.$summaryList.append($list);
			}

			state.$summaryTotal.text($t('Total size charges: %1').replace('%1', formatPrice(total)));
			state.$summary.show();
		}

		function renderMeasurementRows(rows) {
			state.$summaryMeasurements.empty();
			if (!rows.length) {
				state.$summaryMeasurements.hide();
				return;
			}

			var $heading = $('<h4/>', { class: 'measurement-step__summary-heading' }).text($t('Measurements'));
			var $table = $('<table/>', { class: 'measurement-step__summary-table' });
			var $head = $('<thead/>').append(
				$('<tr/>')
					.append($('<th/>').text($t('Measurement')))
					.append($('<th/>').text($t('Value')))
			);
			var $body = $('<tbody/>');

			rows.forEach(function (row) {
				var label = row.label || '';
				var display = row.display_value || row.value || '';
				$body.append(
					$('<tr/>')
						.append($('<th/>').text(label))
						.append($('<td/>').text(display))
				);
			});

			$table.append($head).append($body);
			state.$summaryMeasurements.append($heading).append($table).show();
		}

		function updateHiddenFields(selection) {
			state.$profileId.val(selection.profileId ? String(selection.profileId) : '');
			state.$priceInput.val(String(selection.total || 0));

			var payload = {
				source: selection.source || (selection.profileId ? 'profile' : 'manual'),
				profile_id: selection.profileId || null,
				profile_name: selection.profileName || null,
				profile_type_id: Number(state.config.profileTypeId || 0),
				measurement_type: selection.type,
				measurement_unit: selection.unit,
				measurement_unit_label: selection.unitLabel,
				measurement_type_label: selection.typeLabel,
				measurements: selection.measurements || {},
				summary_rows: buildSummaryRows(selection, true),
				adjustments: selection.adjustments || [],
				total_adjustment: round(selection.total || 0, 2),
				summary_title: selection.summaryTitle || defaultSummaryTitle(selection),
				summary_meta: selection.summaryMeta || buildSummaryMeta(selection),
				field_order: selection.fieldOrder || buildFieldOrder(selection),
				updated_at: selection.updatedAt || null
			};

			state.$payloadInput.val(safeJson(payload));
		}

		function buildSelection(options) {
			options = options || {};
			var measurements = normalizeMeasurements(options.measurements || {});

			var type = normalizeMode(state.activeMode || state.config.measurement.mode || 'body');
			var unit = state.unitValue;

			var summaryRows = buildSummaryRows({
				type: type,
				unit: unit,
				measurements: measurements,
				fieldOrder: buildFieldOrder({ measurements: measurements })
			}, true);

			var adjustments = calculateAdjustments(measurements, type);
			var total = adjustments.reduce(function (carry, item) {
				return carry + (Number(item.amount) || 0);
			}, 0);

			return {
				source: options.source || 'manual',
				type: type,
				typeLabel: getMeasurementTypeLabel(type),
				unit: unit,
				unitLabel: getUnitLabel(unit),
				measurements: measurements,
				adjustments: adjustments,
				summaryRows: summaryRows,
				fieldOrder: buildFieldOrder({ measurements: measurements }),
				total: round(total, 2),
				summaryTitle: defaultSummaryTitle(),
				summaryMeta: buildSummaryMeta({
					type: type,
					unit: unit
				})
			};
		}

		function calculateAdjustments(measurements, mode) {
			var items = [];

			getActiveFields(mode).forEach(function (field) {
				if (!field.price_enabled) {
					updateFieldPricingNote(field.field_id, '');
					return;
				}

				var value = measurements[field.field_id];
				if (typeof value !== 'number') {
					updateFieldPricingNote(field.field_id, '');
					return;
				}

				var tier = matchPricingTier(value, field.pricing_tiers || []);
				if (!tier) {
					updateFieldPricingNote(field.field_id, '');
					return;
				}

				updateFieldPricingNote(field.field_id, tier.tier_label || '');

				var amount = resolveTierAmount(tier);
				if (!amount) {
					return;
				}

				items.push({
					field_id: field.field_id,
					label: field.label || ('Field #' + field.field_id),
					tier: tier.tier_label || '',
					amount: round(amount, 2)
				});
			});

			return items;
		}

		function matchPricingTier(value, tiers) {
			if (!Array.isArray(tiers) || !tiers.length) {
				return null;
			}

			var match = null;
			tiers.some(function (tier) {
				var min = tier.range_min !== undefined && tier.range_min !== null ? coerceNumber(tier.range_min) : null;
				var max = tier.range_max !== undefined && tier.range_max !== null ? coerceNumber(tier.range_max) : null;

				if (min !== null && value < min) {
					return false;
				}

				if (max !== null && value > max) {
					return false;
				}

				match = tier;
				return true;
			});

			return match;
		}

		function resolveTierAmount(tier) {
			if (!tier) {
				return 0;
			}

			var type = (tier.price_type || 'fixed').toLowerCase();
			var amount = tier.price_value !== undefined && tier.price_value !== null
				? coerceNumber(tier.price_value)
				: null;
			if (amount === null && tier.price !== undefined) {
				amount = coerceNumber(tier.price);
			}

			if (amount === null || !isFinite(amount)) {
				return 0;
			}

			if (type === 'percent') {
				var basePrice = Number(state.config.price && state.config.price.base ? state.config.price.base : 0);
				return basePrice * (amount / 100);
			}

			return amount;
		}

		function buildSummaryRows(selection, includeDetails) {
			selection = selection || {};
			var measurements = selection.measurements || {};
			var unitLabel = selection.unitLabel || getUnitLabel(selection.unit) || selection.unit;
			var rows = [];
			var added = {};

			(selection.fieldOrder && selection.fieldOrder.length ? selection.fieldOrder : buildFieldOrder(selection))
				.forEach(function (fieldId) {
					addRow(fieldId);
				});

			Object.keys(measurements).forEach(function (fieldId) {
				addRow(fieldId);
			});

			return rows;

			function addRow(id) {
				var key = String(id);
				if (added[key]) {
					return;
				}

				var value = resolveMeasurementValue(measurements, id);
				if (value === null || value === undefined) {
					return;
				}

				if (typeof value === 'string' && !value.trim()) {
					return;
				}

				var numericId = parseInt(id, 10);
				var definition = state.fieldMap[numericId] || null;
				var label = definition && definition.label
					? definition.label
					: ($t('Field #%1').replace('%1', numericId || id));
				var display = formatMeasurementValue(value, unitLabel);
				if (!display) {
					return;
				}

				var row = {
					field_id: numericId || id,
					label: label,
					display_value: display
				};

				if (includeDetails) {
					row.value = value;
					row.unit_label = unitLabel;
					if (definition && definition.label) {
						row.label = definition.label;
					}
					if (definition && definition.field_type) {
						row.field_type = definition.field_type;
					}
					if (definition && definition.field_code) {
						row.field_code = definition.field_code;
					}
				}

				rows.push(row);
				added[key] = true;
			}
		}

		function buildFieldOrder(selection) {
			selection = selection || {};
			var order = [];

			getActiveFields(selection.type).forEach(function (field) {
				order.push(field.field_id);
			});

			return order;
		}

		function formatMeasurementValue(value, unitLabel) {
			if (value === null || value === undefined) {
				return '';
			}

			if (typeof value === 'number') {
				var rounded = round(value, 2);
				var formatted = Number(rounded).toFixed(2)
					.replace(/\.00$/, '')
					.replace(/(\.[1-9])0$/, '$1');
				return unitLabel ? formatted + ' ' + unitLabel : formatted;
			}

			if (typeof value === 'string') {
				var trimmed = value.trim();
				if (!trimmed.length) {
					return '';
				}
				return unitLabel ? trimmed + ' ' + unitLabel : trimmed;
			}

			return '';
		}

		function buildTailorPayload(selection) {
			if (!selection) {
				return null;
			}

			var rows = selection.summaryRows || buildSummaryRows(selection);
			var descriptionParts = [];

			if (selection.summaryMeta) {
				descriptionParts.push(selection.summaryMeta);
			}

			rows.slice(0, 3).forEach(function (row) {
				if (!row || !row.label) {
					return;
				}
				var value = row.display_value || row.value || '';
				descriptionParts.push(value ? (row.label + ': ' + value) : row.label);
			});

			var items = [{
				code: 'measurement-main',
				label: selection.summaryTitle || defaultSummaryTitle(selection),
				description: descriptionParts.join('\n'),
				qty: '1',
				price: round(selection.total || 0, 2),
				media: {}
			}];

			(selection.adjustments || []).forEach(function (adjustment, index) {
				if (!adjustment) {
					return;
				}
				var amount = Number(adjustment.amount);
				if (!isFinite(amount)) {
					amount = 0;
				}
				var details = [];
				if (adjustment.tier) {
					details.push(String(adjustment.tier));
				}
				if (amount) {
					details.push('+' + formatPrice(amount));
				}
				items.push({
					code: 'measurement-adjustment-' + index,
					label: adjustment.label || $t('Adjustment %1').replace('%1', index + 1),
					description: details.join(' | '),
					qty: '1',
					price: round(amount, 2),
					media: {}
				});
			});

			items = items.filter(function (item) {
				return item && item.label;
			});

			if (!items.length) {
				return null;
			}

			return {
				title: $t('Measurements'),
				items: items,
				total: selection.total || 0
			};
		}

		function pushSummaryToTailor(payload) {
			var form = getProductForm();
			if (!form.length || typeof form.priceTailor !== 'function') {
				return;
			}

			var instance;

			try {
				instance = form.priceTailor('instance');
			} catch (error) {
				instance = null;
			}

			if (!instance) {
				return;
			}

			if (!payload || !payload.items || !payload.items.length) {
				form.priceTailor('setMeasurementSummary', null);
			} else {
				form.priceTailor('setMeasurementSummary', payload);
			}
		}

		function updateEnterButton(hasSelection) {
			state.$enterButton.find('span').text(hasSelection ? state.enterEditLabel : state.enterDefaultLabel);
		}

		function togglePlaceholder(hide) {
			if (!state.$placeholder || !state.$placeholder.length) {
				return;
			}

			state.$placeholder.toggleClass('is-hidden', !!hide);
			state.$placeholder.attr('aria-hidden', hide ? 'true' : 'false');
		}

		function updatePriceBox(amount) {
			var priceBox = getPriceBox();
			if (!priceBox.length) {
				return;
			}

			var value = Number(amount);
			if (!isFinite(value)) {
				value = 0;
			}
			value = round(value, 4);

			var changes = {};
			changes['measurement-adjustment'] = {
				basePrice: { amount: value },
				finalPrice: { amount: value },
				oldPrice: { amount: value }
			};

			priceBox.trigger('updatePrice', changes);
		}

		function setStatus(type, message) {
			if (!state.$status.length) {
				return;
			}

			if (!type || !message) {
				state.$status.empty();
				return;
			}

			state.$status.empty().append($('<div/>', {
				class: 'message message-' + type
			}).text(message));
		}

		function populateTypeOptions() {
			var options = [];
			var seen = {};

			getTypesForOptions().forEach(function (type) {
				var code = normalizeMode(type.value);
				if (!code || seen[code]) {
					return;
				}
				seen[code] = true;
				options.push({
					value: code,
					label: type.label
				});
			});

			state.entry.$typeContainer.empty();

			if (!options.length) {
				state.entry.$typeContainer.append($('<input/>', {
					type: 'hidden',
					name: 'measurement_type',
					value: state.activeMode
				}));
				return;
			}

			options.forEach(function (option, index) {
				var id = 'measurement-type-' + option.value + '-' + index;
				var $item = $('<div/>', { class: 'field choice measurement-profile-form__choice' });
				var $input = $('<input/>', {
					type: 'radio',
					name: 'measurement_type',
					value: option.value,
					id: id,
					class: 'measurement-profile-form__choice-input'
				});
				var $label = $('<label/>', {
					class: 'label measurement-profile-form__choice-label',
					for: id
				}).text(option.label);

				if (option.value === state.activeMode) {
					$input.prop('checked', true);
				}

				$item.append($input).append($label);
				state.entry.$typeContainer.append($item);
			});
		}

		function populateUnitOptions() {
			state.entry.$unitContainer.empty();

			if (!state.measurementUnits.length) {
				state.entry.$unitContainer.append($('<input/>', {
					type: 'hidden',
					name: 'measurement_unit',
					value: state.unitValue
				}));
				return;
			}

			state.measurementUnits.forEach(function (unit, index) {
				var value = unit.code || unit.value;
				if (!value) {
					return;
				}

				var id = 'measurement-unit-' + value + '-' + index;
				var $item = $('<div/>', { class: 'field choice measurement-profile-form__choice' });
				var $input = $('<input/>', {
					type: 'radio',
					name: 'measurement_unit',
					value: value,
					id: id,
					class: 'measurement-profile-form__choice-input',
					'data-role': 'unit-option'
				});
				var $label = $('<label/>', {
					class: 'label measurement-profile-form__choice-label',
					for: id
				}).text(unit.label || value);

				if (value === state.unitValue) {
					$input.prop('checked', true);
				}

				$item.append($input).append($label);
				state.entry.$unitContainer.append($item);
			});
		}

		function renderFieldList() {
			state.entry.fields = {};
			state.entry.$fieldsWrapper.empty();

			var fields = getActiveFields();
			if (!fields.length) {
				state.entry.$fieldsWrapper.append($('<p/>', {
					class: 'measurement-profile-form__empty'
				}).text($t('No measurement fields are available for this selection.')));
				return;
			}

			fields.forEach(function (field) {
				var $field = buildField(field);
				state.entry.$fieldsWrapper.append($field);
			});
		}

		function buildField(field) {
			var fieldId = parseInt(field.field_id, 10);
			var wrapper = $('<div/>', {
				class: 'field measurement-profile-form__field',
				'data-field-id': fieldId
			}).toggleClass('required', !!field.is_required);

			var $label = $('<label/>', {
				class: 'label measurement-profile-form__label',
				for: 'measurement-field-' + fieldId
			}).append($('<span/>', {
				class: 'measurement-profile-form__label-text'
			}).text(field.label || ($t('Field #%1').replace('%1', fieldId))));

			var $control = $('<div/>', { class: 'control measurement-profile-form__control' });
			var $inputWrapper = $('<div/>', { class: 'measurement-profile-form__input-wrapper' });
			var $unitBadge = null;
			var $conversion = null;

			var type = (field.field_type || 'numeric').toLowerCase();
			var $input;

			if (type === 'textarea') {
				$input = $('<textarea/>', {
					id: 'measurement-field-' + fieldId,
					rows: 3,
					class: 'measurement-profile-form__textarea measurement-step__field-control',
					'data-field-id': fieldId,
					'data-field-type': 'textarea'
				});
			} else {
				$input = $('<input/>', {
					type: 'text',
					id: 'measurement-field-' + fieldId,
					class: 'input-text measurement-profile-form__input measurement-step__field-control measurement-step__field-control--numeric',
					autocomplete: 'off',
					'data-field-id': fieldId,
					'data-field-type': 'numeric'
				});

				$unitBadge = $('<span/>', {
					class: 'measurement-profile-form__unit-badge js-field-unit',
					'data-field-id': fieldId
				});

				$conversion = $('<div/>', {
					class: 'measurement-profile-form__conversion js-field-conversion',
					'data-field-id': fieldId,
					style: 'display:none'
				});

				$inputWrapper.addClass('measurement-profile-form__input-wrapper--has-unit');
			}

			$inputWrapper.append($input);
			if ($unitBadge) {
				$inputWrapper.append($unitBadge);
			}
			$control.append($inputWrapper);

			if ($conversion) {
				$control.append($conversion);
			}

			if (field.tooltip_html) {
				$control.append($('<div/>', {
					class: 'measurement-profile-form__note'
				}).html(field.tooltip_html));
			}

			if (field.instruction_html) {
				var $tooltip = $('<div/>', { class: 'field-tooltip toggle measurement-profile-tooltip' });
				var tooltipLabel = $t('View measurement guidance for "%1"').replace('%1', field.label || fieldId);
				var $trigger = $('<span/>', {
					class: 'field-tooltip-action action-help',
					'data-toggle': 'dropdown',
					'aria-haspopup': 'true',
					'aria-expanded': 'false',
					'role': 'button',
					'tabindex': '0',
					'data-field-id': fieldId,
					'aria-label': tooltipLabel,
					title: tooltipLabel
				}).attr('data-mage-init', JSON.stringify({ dropdown: { activeClass: '_active' } }));

				var $content = $('<div/>', {
					class: 'field-tooltip-content',
					'data-target': 'dropdown',
					'aria-hidden': 'true'
				});

				$content.append($('<strong/>', { class: 'measurement-profile-tooltip__title' }).text(field.label || fieldId));
				$content.append($('<div/>', { class: 'measurement-profile-tooltip__body' }).html(field.instruction_html));
				$tooltip.append($trigger).append($content);
				$control.append($tooltip);

				if (mageApply && typeof mageApply.apply === 'function') {
					mageApply.apply($control[0]);
				}

				$trigger.on('click keydown mouseenter focus', function (event) {
					if (event.type === 'keydown') {
						var key = event.key || event.which;
						if (key !== ' ' && key !== 'Spacebar' && key !== 'Enter' && key !== 32 && key !== 13) {
							return;
						}
						event.preventDefault();
					}
					showFieldInstructions(fieldId);
				});
			}

			wrapper.append($label).append($control);
			var $pricingNote = $('<div/>', {
				class: 'measurement-profile-form__pricing-note',
				'data-field-pricing': fieldId
			});
			var $error = $('<div/>', {
				class: 'measurement-profile-form__field-error measurement-step__field-error',
				'data-field-error': fieldId
			});
			wrapper.append($pricingNote).append($error);

			state.entry.fields[fieldId] = {
				field: field,
				$input: $input,
				$error: $error,
				$pricing: $pricingNote,
				$unit: $unitBadge,
				$conversion: $conversion
			};

			return wrapper;
		}

		function renderInstructions() {
			var instructions = state.config.measurement.instructions || {};

			if (instructions.general) {
				state.entry.$generalInstructionContent.html(instructions.general);
				state.entry.$generalInstructionWrapper.show();
			} else {
				state.entry.$generalInstructionContent.empty();
				state.entry.$generalInstructionWrapper.hide();
			}

			var modeHtml = instructions[state.activeMode];
			if (modeHtml) {
				state.entry.$modeInstructionTitle.text(getMeasurementInstructionHeading(state.activeMode));
				state.entry.$modeInstructionContent.html(modeHtml);
				state.entry.$modeInstructionWrapper.show();
				state.entry.$instructionPlaceholderHeading.text(state.entry.$modeInstructionTitle.text()).show();
				state.entry.$instructionPlaceholderContent.html(modeHtml);
			} else {
				state.entry.$modeInstructionTitle.text('');
				state.entry.$modeInstructionContent.empty();
				state.entry.$modeInstructionWrapper.hide();
				resetFieldInstructions();
			}
		}

		function updateUnitIndicators() {
			Object.keys(state.entry.fields).forEach(function (key) {
				var entry = state.entry.fields[key];
				if (!entry.$unit) {
					return;
				}
				entry.$unit.text(getUnitLabel(state.unitValue) || state.unitValue || '');
			});
			updateFieldConversions();
		}

		function updateFieldConversions() {
			Object.keys(state.entry.fields).forEach(function (key) {
				var entry = state.entry.fields[key];
				if (!entry.$conversion) {
					return;
				}

				var raw = (entry.$input.val() || "").trim();
				if (!raw) {
					entry.$conversion.hide().text('');
					return;
				}

				if (!isMetricUnit(state.unitValue)) {
					entry.$conversion.hide().text('');
					return;
				}

				var parsed = parseMeasurement(raw);
				if (parsed === null) {
					entry.$conversion.hide().text('');
					return;
				}

				var inches = parsed / 2.54;
				var formatted = formatDecimal(inches);
				entry.$conversion.text($t('Approx. %1 in').replace('%1', formatted)).show();
			});
		}

		function showFieldInstructions(fieldId) {
			fieldId = parseInt(fieldId, 10);
			var entry = state.entry.fields[fieldId];
			if (!entry || !entry.field || !entry.field.instruction_html) {
				resetFieldInstructions();
				return;
			}

			state.entry.$instructionPlaceholder.hide();
			state.entry.$fieldInstructionTitle.text(entry.field.label || ($t('Field #%1').replace('%1', fieldId)));
			state.entry.$fieldInstructionContent.html(entry.field.instruction_html);
			state.entry.$fieldInstructionWrapper.show();
		}

		function resetFieldInstructions() {
			state.entry.$fieldInstructionTitle.text('');
			state.entry.$fieldInstructionContent.empty();
			state.entry.$fieldInstructionWrapper.hide();
			state.entry.$instructionPlaceholder.show();
		}

		function clearFieldError(fieldId) {
			fieldId = parseInt(fieldId, 10);
			var entry = state.entry.fields[fieldId];
			if (!entry) {
				return;
			}
			entry.$error.text('').hide();
		}

		function showFieldError(fieldId, message) {
			fieldId = parseInt(fieldId, 10);
			var entry = state.entry.fields[fieldId];
			if (!entry) {
				return;
			}
			entry.$error.text(message).show();
		}

		function updateFieldPricingNote(fieldId, text) {
			fieldId = parseInt(fieldId, 10);
			var entry = state.entry.fields[fieldId];
			if (!entry || !entry.$pricing) {
				return;
			}
			if (text) {
				entry.$pricing.text(text).show();
			} else {
				entry.$pricing.text('').hide();
			}
		}

		function normalizeNumericInput($input) {
			var fieldId = $input.data('field-id');
			var parsed = parseMeasurement($input.val());

			if (parsed === null) {
				showFieldError(fieldId, $t('Enter a valid number.'));
				return;
			}

			$input.val(formatDecimal(parsed));
			clearFieldError(fieldId);
			updateFieldConversions();
		}

		function renderProfileModal(profiles) {
			var $modal = ensureProfileModal();
			var $body = $('<div/>', { class: 'measurement-step__profiles' });

			if (!profiles.length) {
				$body.append($('<p/>', { class: 'measurement-step__profiles-empty' }).text($t('No saved measurements found.')));
			} else {
				profiles.forEach(function (profile) {
					$body.append(buildProfileCard(profile));
				});
			}

			state.$modalContainer.empty().append($body);
			$modal.modal('openModal');
		}

		function buildProfileCard(profile) {
			var $card = $('<div/>', { class: 'measurement-step__profile-card' });
			$card.append($('<h3/>').text(profile.profile_name || ('#' + profile.profile_id)));
			$card.append($('<p/>').text(buildProfileMeta(profile)));

			var $button = $('<button/>', {
				type: 'button',
				class: 'action primary'
			}).text($t('Use these measurements'));

			$button.on('click', function () {
				applyProfile(profile, $button);
			});

			$card.append($button);
			return $card;
		}

		function applyProfile(profile, $trigger) {
			if (!profile) {
				return;
			}

			setProfileButtonState($trigger, true);

			var request = ensureProfileFieldDefinitions(profile);
			request.then(function (details) {
				if (details && details.profile) {
					profile.profile_name = details.profile.profile_name || profile.profile_name;
				}

				var measurements = normalizeMeasurements(profile.measurements || profile.measurements_data || {});
				profile.measurements = measurements;

				var selection = {
					source: 'profile',
					profileId: profile.profile_id,
					profileName: profile.profile_name || null,
					type: normalizeMode(profile.measurement_type || state.activeMode),
					typeLabel: profile.measurement_type_label || getMeasurementTypeLabel(profile.measurement_type || state.activeMode),
					unit: profile.measurement_unit || state.unitValue,
					unitLabel: profile.measurement_unit_label || getUnitLabel(profile.measurement_unit || state.unitValue),
					measurements: measurements,
					summaryRows: profile.summary_rows || null,
					fieldOrder: profile.field_order || [],
					total: Number(profile.total_adjustment) || 0,
					summaryTitle: profile.profile_name || defaultSummaryTitle(),
					summaryMeta: buildProfileMeta(profile),
					updatedAt: profile.updated_at || null
				};

				applySelection(selection);
				if (state.profileModal) {
					state.profileModal.modal('closeModal');
				}
				setStatus();
				setProfileButtonState($trigger, false);
			}).catch(function (error) {
				setStatus('error', error && error.message ? error.message : $t('Unable to load measurement details.'));
				setProfileButtonState($trigger, false);
			});
		}

		function buildProfileMeta(profile) {
			var parts = [];
			if (profile.measurement_type) {
				parts.push(getMeasurementTypeLabel(profile.measurement_type));
			}
			if (profile.measurement_unit) {
				parts.push(getUnitLabel(profile.measurement_unit));
			}
			return parts.join(' · ');
		}

		function ensureProfileModal() {
			if (state.profileModal && state.profileModal.modal) {
				return state.profileModal;
			}

			state.profileModal = state.$modalContainer;
			modal({
				title: $t('Select measurements'),
				modalClass: 'measurement-step__profiles-modal',
				zIndex: 1200,
				buttons: [{
					text: $t('Close'),
					class: 'action secondary',
					click: function () {
						this.closeModal();
					}
				}]
			}, state.profileModal);

			return state.profileModal;
		}

		function showProfilesLoading() {
			var $modal = ensureProfileModal();
			var $loading = $('<div/>', { class: 'measurement-step__profiles measurement-step__profiles--loading' });
			var $mask = $('<div/>', { class: 'loading-mask', style: 'display:block' }).append($('<div/>', { class: 'loader' }));
			$loading.append($mask);
			$loading.append($('<p/>', { class: 'measurement-step__profiles-message' }).text($t('Loading saved measurements…')));
			state.$modalContainer.empty().append($loading);
			$modal.modal('openModal');
		}

		function showProfileModalError(message) {
			var $modal = ensureProfileModal();
			var $error = $('<div/>', { class: 'measurement-step__profiles measurement-step__profiles--error' });
			$error.append($('<div/>', { class: 'message error' }).text(message));
			state.$modalContainer.empty().append($error);
			$modal.modal('openModal');
		}

		function normalizeProfiles(list) {
			return (Array.isArray(list) ? list : []).map(function (profile) {
				if (!profile) {
					return null;
				}

				var id = Number(profile.profile_id || profile.id || 0);
				var name = profile.profile_name || profile.name || '';
				var type = normalizeMode(profile.measurement_type || state.activeMode);
				var unit = profile.measurement_unit || state.unitValue;
				var measurements = normalizeMeasurements(profile.measurements || profile.measurements_data || {});
				var fieldOrder = Array.isArray(profile.field_order) ? profile.field_order.slice() : [];
				var typeLabel = profile.measurement_type_label || getMeasurementTypeLabel(type);
				var unitLabel = profile.measurement_unit_label || getUnitLabel(unit);
				var summaryRows = Array.isArray(profile.summary_rows) && profile.summary_rows.length
					? profile.summary_rows
					: buildSummaryRows({
						measurements: measurements,
						unit: unit,
						unitLabel: unitLabel,
						type: type,
						fieldOrder: fieldOrder
					}, true);
				var profileTypeId = Number(profile.profile_type_id || profile.profile_type || state.config.profileTypeId || 0);
				var profileTypeLabel = profile.profile_type_label || state.config.profileTypeLabel || '';

				var measurementLabels = {};
				summaryRows.forEach(function (row) {
					if (row && row.field_id) {
						measurementLabels[row.field_id] = row.label;
					}
				});

				return {
					profile_id: id,
					profile_name: name,
					profile_type_id: profileTypeId,
					profile_type_label: profileTypeLabel,
					measurements: measurements,
					measurement_type: type,
					measurement_type_label: typeLabel,
					measurement_unit: unit,
					measurement_unit_label: unitLabel,
					field_order: fieldOrder,
					summary_rows: summaryRows,
					raw_summary_rows: Array.isArray(profile.summary_rows) ? profile.summary_rows : null,
					total_adjustment: profile.total_adjustment !== undefined ? profile.total_adjustment : null,
					updated_at: profile.updated_at || null,
					measurement_labels: measurementLabels
				};
			}).filter(Boolean);
		}

		function ensureProfileFieldDefinitions(profile) {
			profile = profile || {};
			var profileId = Number(profile.profile_id || 0);
			var profileTypeId = Number(profile.profile_type_id || state.config.profileTypeId || 0);
			var type = normalizeMode(profile.measurement_type || state.activeMode);
			var cacheKey = profileTypeId + ':' + type;
			var cachedFields = state.fieldDefinitionCache[cacheKey];
			var cachedDetails = profileId ? state.profileDetailsCache[profileId] : null;

			if (cachedDetails && cachedDetails.type === type) {
				registerFieldDefinitions(type, cachedDetails.fields);
				enrichProfileWithFieldDefinitions(profile, cachedDetails.fields, type, cachedDetails.profileTypeId, cachedDetails.profile);
				return Promise.resolve({ fields: cachedDetails.fields, profile: cachedDetails.profile });
			}

			if (!profileId && cachedFields) {
				registerFieldDefinitions(type, cachedFields);
				enrichProfileWithFieldDefinitions(profile, cachedFields, type, profileTypeId, null);
				return Promise.resolve({ fields: cachedFields, profile: null });
			}

			if (!state.config.urls || !state.config.urls.form || !profileId) {
				var fallbackFields = cachedFields || state.fieldsByType[type] || state.fieldsByType[state.config.measurement.mode] || [];
				registerFieldDefinitions(type, fallbackFields);
				enrichProfileWithFieldDefinitions(profile, fallbackFields, type, profileTypeId, null);
				return Promise.resolve({ fields: fallbackFields, profile: null });
			}

			var params = {
				profile_id: profileId,
				measurement_type: type
			};

			if (profileTypeId) {
				params.profile_type_id = profileTypeId;
			}

			var url = buildUrlWithParams(state.config.urls.form, params);

			return toPromise(storage.get(url, false))
				.then(function (response) {
					if (!response || !response.success) {
						throw new Error(response && response.message ? response.message : $t('Unable to load measurement definitions.'));
					}

					var resolvedProfile = response.profile || {};
					var resolvedType = normalizeMode(response.measurement_type || resolvedProfile.measurement_type || type);
					var fields = prepareFields(response.fields || []);
					var resolvedProfileTypeId = Number(response.profile_type_id || resolvedProfile.profile_type_id || profileTypeId || 0);
					var resolvedCacheKey = resolvedProfileTypeId + ':' + resolvedType;

					state.fieldDefinitionCache[resolvedCacheKey] = fields;
					state.fieldDefinitionCache[cacheKey] = fields;
					state.profileDetailsCache[profileId] = {
						fields: fields,
						profile: resolvedProfile,
						type: resolvedType,
						profileTypeId: resolvedProfileTypeId
					};

					registerFieldDefinitions(resolvedType, fields);
					enrichProfileWithFieldDefinitions(profile, fields, resolvedType, resolvedProfileTypeId, resolvedProfile);

					return {
						fields: fields,
						profile: resolvedProfile
					};
				});
		}

		function enrichProfileWithFieldDefinitions(profile, fields, type, profileTypeId, sourceProfile) {
			if (!profile) {
				return;
			}

			var normalizedType = normalizeMode((sourceProfile && sourceProfile.measurement_type) || profile.measurement_type || type || state.activeMode);
			profile.measurement_type = normalizedType;
			profile.measurement_type_label = getMeasurementTypeLabel(normalizedType);
			if (profileTypeId) {
				profile.profile_type_id = Number(profileTypeId);
			}
			if (sourceProfile && sourceProfile.profile_name) {
				profile.profile_name = sourceProfile.profile_name;
			}
			if (sourceProfile && sourceProfile.updated_at) {
				profile.updated_at = sourceProfile.updated_at;
			}
			if (sourceProfile && sourceProfile.total_adjustment !== undefined) {
				profile.total_adjustment = sourceProfile.total_adjustment;
			}
			if (!profile.profile_type_label) {
				profile.profile_type_label = state.config.profileTypeLabel || '';
			}

			var preparedFields = Array.isArray(fields) ? fields : [];
			if (preparedFields.length) {
				profile.field_order = preparedFields.map(function (field) {
					var id = parseInt(field.field_id, 10);
					return isNaN(id) ? null : id;
				}).filter(function (id) {
					return id !== null;
				});
			}

			var measurementsData = (sourceProfile && sourceProfile.measurements_data) || profile.measurements_data || profile.measurements || {};
			profile.measurements_data = measurementsData;
			profile.measurements = normalizeMeasurements(measurementsData);

			if (sourceProfile && sourceProfile.measurement_unit) {
				profile.measurement_unit = sourceProfile.measurement_unit;
			}
			var unit = profile.measurement_unit || state.unitValue;
			if (sourceProfile && sourceProfile.measurement_unit_label) {
				profile.measurement_unit_label = sourceProfile.measurement_unit_label;
			}
			var unitLabel = profile.measurement_unit_label || getUnitLabel(unit);
			profile.measurement_unit = unit;
			profile.measurement_unit_label = unitLabel;

			var summarySelection = {
				measurements: profile.measurements || {},
				unit: unit,
				unitLabel: unitLabel,
				type: normalizedType,
				fieldOrder: profile.field_order || []
			};
			profile.summary_rows = buildSummaryRows(summarySelection, true);

			var measurementLabels = {};
			(profile.summary_rows || []).forEach(function (row) {
				if (row && row.field_id) {
					measurementLabels[row.field_id] = row.label;
				}
			});
			profile.measurement_labels = measurementLabels;
		}

		function registerFieldDefinitions(type, fields) {
			var normalized = normalizeMode(type || state.config.measurement.mode);
			var prepared = prepareFields(fields || []);

			state.fieldsByType[normalized] = prepared;

			if (normalizeMode(state.config.measurement.mode) === normalized) {
				state.fields = prepared.slice();
			}

			state.fieldMap = buildFieldMap(state.fieldsByType, state.fields);

			if (state.entry && state.entry.ready && normalizeMode(state.activeMode) === normalized) {
				renderFieldList();
				renderInstructions();
				updateUnitIndicators();
			}
		}

		function buildUrlWithParams(url, params) {
			var query = $.param(params || {});
			if (!query) {
				return url;
			}
			return url + (url.indexOf('?') === -1 ? '?' : '&') + query;
		}

		function setLoadButtonState(pending) {
			state.$loadButton.prop('disabled', !!pending).toggleClass('is-loading', !!pending);
		}

		function setProfileButtonState($button, pending) {
			if (!$button || !$button.length) {
				return;
			}

			$button.prop('disabled', !!pending).toggleClass('is-loading', !!pending);
			if (pending) {
				$button.attr('aria-busy', 'true');
			} else {
				$button.removeAttr('aria-busy');
			}
		}

		function switchMeasurementMode(mode) {
			var normalized = normalizeMode(mode);
			if (!normalized) {
				normalized = 'body';
			}

			if (normalized === state.activeMode) {
				return;
			}

			state.activeMode = normalized;
			renderFieldList();
			renderInstructions();
			updateUnitIndicators();
			syncTypeOptions(normalized);
		}

		function syncTypeOptions(value) {
			state.entry.$typeContainer.find('input[name="measurement_type"]').each(function () {
				var $radio = $(this);
				$radio.prop('checked', normalizeMode($radio.val()) === value);
			});
		}

		function syncUnitOptions(value) {
			state.entry.$unitContainer.find('[data-role="unit-option"]').each(function () {
				var $option = $(this);
				if ($option.is(':radio')) {
					$option.prop('checked', $option.val() === value);
				} else {
					$option.val(value);
				}
			});
		}

		function bindTailorIntegration() {
			var form = getProductForm();
			if (!form.length) {
				return;
			}

			form.on('priceTailorInitialized', function () {
				pushSummaryToTailor(buildTailorPayload(state.selection));
			});
		}

		function findAddToCartButtons() {
			var form = getProductForm();
			if (!form.length) {
				return $();
			}

			var buttons = form.find('button[type="submit"], button.action.tocart, [data-role="tocart"]');
			return buttons.filter(function () {
				return $(this).closest('form')[0] === form[0];
			});
		}

		function setAddToCartEnabled(enabled) {
			enabled = !!enabled;
			state.addToCartEnabled = enabled;
			var buttons = state.$addToCartButtons && state.$addToCartButtons.length
				? state.$addToCartButtons
				: (state.$addToCartButtons = findAddToCartButtons());

			if (!buttons.length) {
				return;
			}

			buttons.prop('disabled', !enabled);
			buttons.toggleClass('is-disabled', !enabled);
			buttons.attr('aria-disabled', enabled ? 'false' : 'true');
		}

		function refreshAddToCartEnabled() {
			setAddToCartEnabled(hasActiveSelection());
		}

		function bindFormValidation() {
			if (state.formValidationBound) {
				return;
			}

			var form = getProductForm();
			if (!form.length) {
				return;
			}

			state.formValidationBound = true;

			form.on('submit.measurementIntegration', function (event) {
				if (hasActiveSelection()) {
					return;
				}

				event.preventDefault();
				event.stopImmediatePropagation();
				event.stopPropagation();

				setAddToCartEnabled(false);
				setStatus('error', $t('Please add your measurements before adding to cart.'));
				state.$enterButton.focus();
				ensureEntryPanel();
				showEntryPanel();
			});

			if (!state.nativeFormSubmit && form.length) {
				state.nativeFormSubmit = form[0].submit;
				form[0].submit = function () {
					if (!hasActiveSelection()) {
						setAddToCartEnabled(false);
						setStatus('error', $t('Please add your measurements before adding to cart.'));
						state.$enterButton.focus();
						ensureEntryPanel();
						showEntryPanel();
						return false;
					}

					return state.nativeFormSubmit.apply(this, arguments);
				};
			}
		}

		function getProductForm() {
			if (state.$productForm && state.$productForm.length) {
				return state.$productForm;
			}

			var form = $('#product_addtocart_form');
			if (!form.length) {
				form = state.$root.closest('#product_addtocart_form');
			}
			if (!form.length) {
				form = state.$root.closest('form');
			}
			state.$productForm = form;

			return state.$productForm;
		}

		function getPriceBox() {
			if (state.$priceBox && state.$priceBox.length) {
				return state.$priceBox;
			}

			var form = getProductForm();
			var priceBox = form.find('[data-role="priceBox"]');
			if (!priceBox.length) {
				priceBox = form.find('.price-box');
			}
			if (!priceBox.length) {
				priceBox = $('[data-role="priceBox"]');
			}

			state.$priceBox = priceBox;
			return state.$priceBox;
		}

		function prepareFields(list) {
			var items = Array.isArray(list) ? list.slice() : [];
			items = items.map(function (field) {
				return normalizeFieldDefinition(field);
			});
			items.sort(function (a, b) {
				return (a.sort_order || 0) - (b.sort_order || 0);
			});
			return items;
		}

		function normalizeFieldDefinition(field) {
			if (!field) {
				return field;
			}

			if (field.field_id !== undefined) {
				var numericId = parseInt(field.field_id, 10);
				if (!isNaN(numericId)) {
					field.field_id = numericId;
				}
			}

			if (field.sort_order !== undefined) {
				var order = parseInt(field.sort_order, 10);
				field.sort_order = isNaN(order) ? 0 : order;
			}

			if (field.is_required !== undefined) {
				field.is_required = coerceBoolean(field.is_required);
			}

			if (field.price_enabled !== undefined) {
				field.price_enabled = coerceBoolean(field.price_enabled);
			}

			if (Array.isArray(field.pricing_tiers)) {
				field.pricing_tiers = field.pricing_tiers.map(function (tier) {
					if (!tier) {
						return tier;
					}

					if (tier.range_min === undefined && tier.min_value !== undefined) {
						tier.range_min = coerceNumber(tier.min_value);
					}
					if (tier.range_max === undefined && tier.max_value !== undefined) {
						tier.range_max = coerceNumber(tier.max_value);
					}
					if (tier.range_min !== undefined) {
						tier.range_min = coerceNumber(tier.range_min);
					}
					if (tier.range_max !== undefined) {
						tier.range_max = coerceNumber(tier.range_max);
					}
					if (tier.price_value !== undefined) {
						tier.price_value = coerceNumber(tier.price_value);
					}
					if (tier.price !== undefined) {
						tier.price = coerceNumber(tier.price);
					}

					return tier;
				});
			}

			return field;
		}

		function coerceBoolean(value) {
			return value === true || value === 1 || value === '1' || value === 'true';
		}

		function coerceNumber(value) {
			if (value === null || value === undefined || value === '') {
				return null;
			}
			var numeric = Number(value);
			return isFinite(numeric) ? numeric : null;
		}

		function getActiveFields(mode) {
			var key = normalizeMode(mode || state.activeMode);
			var fields = state.fieldsByType[key];
			if (!Array.isArray(fields) || !fields.length) {
				fields = state.fieldsByType[state.config.measurement.mode] || state.fields;
			}
			return Array.isArray(fields) ? fields : [];
		}

		function buildFieldMap(byType, fallback) {
			var map = {};
			Object.keys(byType || {}).forEach(function (mode) {
				(byType[mode] || []).forEach(function (field) {
					var id = parseInt(field.field_id, 10);
					if (id) {
						map[id] = field;
					}
				});
			});
			(fallback || []).forEach(function (field) {
				var id = parseInt(field.field_id, 10);
				if (id) {
					map[id] = field;
				}
			});
			return map;
		}
		function resolveDefaultUnit() {
			var profileType = state.config.measurement.profileType;
			if (profileType && profileType.default_measurement_unit) {
				return profileType.default_measurement_unit;
			}
			if (state.measurementUnits.length) {
				return state.measurementUnits[0].code || state.measurementUnits[0].value;
			}
			return '';
		}

		function normalizeMode(mode) {
			if (!mode) {
				return 'body';
			}
			var value = String(mode).toLowerCase();
			return value === 'garment' ? 'garment' : 'body';
		}

		function getTypesForOptions() {
			var base = [];
			if (state.measurementTypes.length) {
				base = state.measurementTypes.map(function (type) {
					return {
						value: type.code || type.value,
						label: type.label || type.code || type.value
					};
				});
			}

			Object.keys(state.fieldsByType).forEach(function (mode) {
				var label = getMeasurementTypeLabel(mode);
				base.push({ value: mode, label: label });
			});

			return base;
		}

		function getMeasurementTypeLabel(code) {
			code = normalizeMode(code || state.activeMode);
			if (state.typeLabels[code]) {
				return state.typeLabels[code];
			}
			return code.charAt(0).toUpperCase() + code.slice(1);
		}

		function getMeasurementInstructionHeading(mode) {
			return mode === 'garment'
				? $t('Garment Measurement Guidance')
				: $t('Body Measurement Guidance');
		}

		function getUnitLabel(code) {
			if (!code) {
				return '';
			}
			var key = String(code);
			return state.unitLabels[key] || key;
		}

		function defaultSummaryTitle(selection) {
			selection = selection || {};
			if (selection.profileName) {
				return selection.profileName;
			}
			return $t('Custom measurements');
		}

		function buildSummaryMeta(selection) {
			selection = selection || {};
			var parts = [];
			var type = selection.type || state.activeMode;
			if (type) {
				parts.push(getMeasurementTypeLabel(type));
			}
			if (selection.unit || state.unitValue) {
				parts.push(getUnitLabel(selection.unit || state.unitValue));
			}
			return parts.join(' · ');
		}

		function parseMeasurement(value) {
			if (value === null || value === undefined) {
				return null;
			}

			var trimmed = (String(value).trim());
			if (!trimmed) {
				return null;
			}

			var numeric = Number(trimmed.replace(/,/g, ''));
			if (!isNaN(numeric)) {
				return round(numeric, DEFAULT_PRECISION);
			}

			var match = trimmed.match(FRACTION_REGEX);
			if (match) {
				var whole = match[1] ? Number(match[1]) : 0;
				var fraction = match[2] ? match[2].split('/') : null;
				var fractionValue = 0;
				if (fraction && fraction.length === 2) {
					var numerator = Number(fraction[0]);
					var denominator = Number(fraction[1]);
					if (isFinite(numerator) && isFinite(denominator) && denominator !== 0) {
						fractionValue = numerator / denominator;
					}
				}
				return round(whole + fractionValue, DEFAULT_PRECISION);
			}

			return null;
		}

		function formatDecimal(value) {
			if (value === null || value === undefined || isNaN(value)) {
				return '';
			}
			return round(Number(value), DEFAULT_PRECISION).toFixed(DEFAULT_PRECISION)
				.replace(/\.00$/, '')
				.replace(/(\.[1-9])0$/, '$1');
		}

		function normalizeMeasurements(raw) {
			if (!raw) {
				return {};
			}

			if (typeof raw === 'string') {
				try {
					raw = JSON.parse(raw);
				} catch (error) {
					raw = {};
				}
			}

			var normalized = {};
			Object.keys(raw).forEach(function (key) {
				var value = raw[key];
				var id = parseInt(key, 10);
				var definition = state.fieldMap[id];

				if (definition && (definition.field_type || 'numeric').toLowerCase() === 'numeric') {
					if (typeof value === 'number') {
						normalized[id] = value;
					} else {
						var parsed = parseMeasurement(value);
						if (parsed !== null) {
							normalized[id] = parsed;
						}
					}
				} else {
					normalized[id] = value;
				}
			});

			return normalized;
		}

		function selectionFromPayload(payload) {
			if (!payload) {
				return null;
			}

			return {
				source: payload.source || 'manual',
				profileId: payload.profile_id || null,
				profileName: payload.profile_name || null,
				type: normalizeMode(payload.measurement_type || state.activeMode),
				typeLabel: payload.measurement_type_label || getMeasurementTypeLabel(payload.measurement_type),
				unit: payload.measurement_unit || state.unitValue,
				unitLabel: payload.measurement_unit_label || getUnitLabel(payload.measurement_unit),
				measurements: normalizeMeasurements(payload.measurements),
				summaryRows: payload.summary_rows || [],
				adjustments: Array.isArray(payload.adjustments) ? payload.adjustments : [],
				fieldOrder: payload.field_order || [],
				total: payload.total_adjustment !== undefined ? Number(payload.total_adjustment) : 0,
				summaryTitle: payload.summary_title || null,
				summaryMeta: payload.summary_meta || null,
				updatedAt: payload.updated_at || null
			};
		}

		function hasActiveSelection() {
			if (selectionHasValues(state.selection)) {
				return true;
			}

			var raw = '';
			if (state.$payloadInput && state.$payloadInput.length) {
				raw = (state.$payloadInput.val() || '').trim();
			}
			if (!raw) {
				return false;
			}

			var payload;
			try {
				payload = JSON.parse(raw);
			} catch (error) {
				payload = null;
			}

			if (selectionHasValues(payload)) {
				return true;
			}

			var selection = selectionFromPayload(payload);
			return selectionHasValues(selection);
		}

		function selectionHasValues(data) {
			if (!data) {
				return false;
			}

			var rows = data.summaryRows || data.summary_rows;
			if (Array.isArray(rows) && rows.length) {
				return true;
			}

			var measurements = data.measurements || data.measurements_data || {};
			var keys = Object.keys(measurements);
			if (!keys.length) {
				return false;
			}

			return keys.some(function (key) {
				if (!Object.prototype.hasOwnProperty.call(measurements, key)) {
					return false;
				}
				var value = measurements[key];
				if (value === null || value === undefined) {
					return false;
				}
				if (typeof value === 'string') {
					return value.trim().length > 0;
				}
				return true;
			});
		}

		function formatPrice(amount) {
			return priceUtils.formatPrice(amount || 0, state.config.price && state.config.price.format ? state.config.price.format : {});
		}

		function formatDate(input) {
			if (!input) {
				return '';
			}

			var date = new Date(input);
			if (isNaN(date.getTime())) {
				return '';
			}

			return date.toLocaleDateString();
		}

		function round(value, precision) {
			var factor = Math.pow(10, precision || 0);
			return Math.round(Number(value) * factor) / factor;
		}

		function safeJson(value) {
			try {
				return JSON.stringify(value);
			} catch (error) {
				return '""';
			}
		}

		function isMetricUnit(code) {
			if (!code) {
				return false;
			}
			var normalized = String(code).toLowerCase();
			return normalized === 'cm' || normalized === 'centimeter' || normalized === 'centimetre';
		}

		function getFormKey() {
			return getCookie('form_key') || (typeof window !== 'undefined' ? window.FORM_KEY : null);
		}

		function updateFormKey(value) {
			if (!value) {
				return;
			}
			if (typeof window !== 'undefined') {
				window.FORM_KEY = value;
			}
			setCookie('form_key', value);
			$('input[name="form_key"]').val(value);
		}

		function getCookie(name) {
			if (cookies && typeof cookies.get === 'function') {
				return cookies.get(name);
			}
			if (typeof document === 'undefined') {
				return null;
			}
			var match = document.cookie.match(new RegExp('(?:^|; )' + encodeURIComponent(name) + '=([^;]*)'));
			return match ? decodeURIComponent(match[1]) : null;
		}

		function setCookie(name, value) {
			if (cookies && typeof cookies.set === 'function') {
				cookies.set(name, value);
				return;
			}
			if (typeof document === 'undefined') {
				return;
			}
			document.cookie = encodeURIComponent(name) + '=' + encodeURIComponent(value || '') + '; path=/';
		}

		function findPlaceholder() {
			var placeholder = state.$root.closest('[data-role="wizard-panel"]').find('[data-role="measurement-placeholder"]').first();
			if (!placeholder.length) {
				placeholder = state.$root.closest('[data-role="wizard-panel-body"]').find('[data-role="measurement-placeholder"]').first();
			}
			return placeholder;
		}

		function toPromise(request) {
			if (!request) {
				return Promise.reject(new Error('Request unavailable'));
			}
			if (typeof request.then === 'function') {
				return request;
			}
			return new Promise(function (resolve, reject) {
				var resolved = false;
				if (typeof request.done === 'function') {
					request.done(function (response) {
						resolved = true;
						resolve(response);
					});
				}
				if (typeof request.fail === 'function') {
					request.fail(function (error) {
						reject(error);
					});
				} else if (typeof request.always === 'function') {
					request.always(function (response) {
						if (!resolved) {
							resolve(response);
						}
					});
				}
			});
		}
	};
});
