<?php

namespace Noesis\StripeGraphQLFix\Plugin\Sales\Model\Service;

use Magento\Framework\GraphQl\Exception\GraphQlInputException;

/**
 * Replaces StripeIntegration\Payments\Plugin\Sales\Model\Service\OrderService
 * via <preference> in di.xml (same approach as ScandiPWA/stripe-graphql).
 *
 * Key fix: when $order->getId() is set inside the catch block, the order was
 * already persisted to the DB by OrderService::place(). The exception came from
 * postProcess() (e.g. "An ID is needed" during payment-transaction save on
 * Path B after 3DS). Return the saved order so
 * QuoteManagement::submitQuote() can commit its transaction normally instead of
 * rolling it back.
 */
class OrderService
{
    private $helper;
    private $config;
    private $helperFactory;
    private $quoteHelper;
    private $webhookEventCollectionFactory;
    private $paymentMethodHelper;
    private $loggerHelper;
    private $orderHelper;
    private $paymentState;
    private $checkoutCrashHelper;
    private $radarHelper;

    public function __construct(
        \StripeIntegration\Payments\Helper\Quote $quoteHelper,
        \StripeIntegration\Payments\Helper\Order $orderHelper,
        \StripeIntegration\Payments\Helper\GenericFactory $helperFactory,
        \StripeIntegration\Payments\Helper\PaymentMethod $paymentMethodHelper,
        \StripeIntegration\Payments\Helper\Logger $loggerHelper,
        \StripeIntegration\Payments\Helper\CheckoutCrash $checkoutCrashHelper,
        \StripeIntegration\Payments\Helper\Radar $radarHelper,
        \StripeIntegration\Payments\Model\Config $config,
        \StripeIntegration\Payments\Model\Order\PaymentState $paymentState,
        \StripeIntegration\Payments\Model\ResourceModel\WebhookEvent\CollectionFactory $webhookEventCollectionFactory
    ) {
        $this->quoteHelper = $quoteHelper;
        $this->orderHelper = $orderHelper;
        $this->helperFactory = $helperFactory;
        $this->paymentMethodHelper = $paymentMethodHelper;
        $this->loggerHelper = $loggerHelper;
        $this->checkoutCrashHelper = $checkoutCrashHelper;
        $this->radarHelper = $radarHelper;
        $this->config = $config;
        $this->paymentState = $paymentState;
        $this->webhookEventCollectionFactory = $webhookEventCollectionFactory;
    }

    public function aroundPlace($subject, \Closure $proceed, $order)
    {
        try
        {
            if (!empty($order) && !empty($order->getQuoteId()))
            {
                $this->quoteHelper->quoteId = $order->getQuoteId();
            }

            $savedOrder = $proceed($order);

            return $this->postProcess($savedOrder);
        }
        catch (\Exception $e)
        {
            $helper = $this->getHelper();
            $msg = $e->getMessage();

            if ($order->getId())
            {
                // The order was already saved to the DB by OrderService::place().
                // The exception came from postProcess() -- "An ID is needed" during
                // payment transaction save on Path B (manual 3DS second placeOrder).
                // Return the saved order so QuoteManagement::submitQuote() commits normally.
                return isset($savedOrder) ? $savedOrder : $order;
            }

            if ($this->isAuthenticationRequiredMessage($msg))
            {
                // Path B step 1: PI requires 3DS, no order created yet.
                // Re-throw so the frontend can extract the client_secret.
                throw $e;
            }

            if ($this->paymentState->isPaid() && empty($savedOrder))
            {
                $this->checkoutCrashHelper->log($this->paymentState, $e)
                    ->notifyAdmin($this->paymentState, $e)
                    ->deactivateCart();

                $helper->logError($e->getMessage(), $e->getTraceAsString());

                throw $e;
            }

            // Payment failed errors
            return $helper->throwError($e->getMessage(), $e);
        }
    }

    public function postProcess($order)
    {
        if (strstr($order->getPayment()->getMethod(), "stripe_") !== false)
        {
            try
            {
                $this->paymentMethodHelper->saveOrderPaymentMethodById(
                    $order,
                    $order->getPayment()->getAdditionalInformation("token")
                );
            }
            catch (\Exception $e)
            {
                $this->loggerHelper->logError(
                    "Failed to save order payment method: " . $e->getMessage(),
                    $e->getTraceAsString()
                );
            }

            try
            {
                $this->radarHelper->setOrderRiskData($order);
            }
            catch (\Exception $e)
            {
                $this->loggerHelper->logError(
                    "Failed to save order risk data: " . $e->getMessage(),
                    $e->getTraceAsString()
                );
            }

            $this->orderHelper->saveOrder($order);
        }

        $helper = $this->getHelper();
        switch ($order->getPayment()->getMethod())
        {
            case "stripe_payments_invoice":
                $comment = __("A payment is pending for this order.");
                $helper->setOrderState($order, \Magento\Sales\Model\Order::STATE_PENDING_PAYMENT, $comment);
                $this->orderHelper->saveOrder($order);
                break;

            case "stripe_payments":
            case "stripe_payments_express":
                if ($transactionId = $order->getPayment()->getAdditionalInformation("server_side_transaction_id"))
                {
                    $events = $this->webhookEventCollectionFactory->create()
                        ->getEarlyEventsForPaymentIntentId($transactionId, [
                            'charge.succeeded',
                            'invoice.payment_succeeded',
                            'setup_intent.succeeded'
                        ]);

                    foreach ($events as $eventModel)
                    {
                        try
                        {
                            $eventModel->process($this->config->getStripeClient());
                        }
                        catch (\Exception $e)
                        {
                            $eventModel->refresh()->setLastErrorFromException($e);
                        }
                    }
                }
                break;

            default:
                break;
        }

        return $order;
    }

    // Cannot use the helper method due to circular dependency (same reason as ScandiPWA).
    private function isAuthenticationRequiredMessage($message)
    {
        return strpos($message, "Authentication Required: ") === 0;
    }

    protected function getHelper()
    {
        if (!isset($this->helper))
        {
            $this->helper = $this->helperFactory->create();
        }

        return $this->helper;
    }
}
